<?php declare(strict_types = 1);

namespace EshopProductsComparison\FrontModule\Model\Export;

use Contributte\Translation\Translator;
use Core\Model\Dao\Site;
use Core\Model\Dao\SiteDomain;
use Core\Model\Entities\Country;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Strings;
use Core\Model\Settings;
use Currency\Model\Entities\Currency;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureTexts;
use EshopCatalog\Model\Entities\FeatureValue;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use EshopCatalog\Model\Entities\Tag;
use EshopGifts\FrontModule\Model\OrderGifts;
use EshopOrders\Model\PaymentSpeditions;
use EshopProductsComparison\FrontModule\Model\Provider\IXmlExportService;
use EshopProductsComparison\Model\Helpers\ExportEnums;
use MeasuringCodes\FrontModule\Model\TypesList;
use Nette\DI\Container;
use Nette\Utils\FileSystem;

abstract class XmlService implements IXmlExportService
{
	/** @var resource|null */
	protected $fileHandle = null;

	/** @var resource|null */
	protected $fileHandleStock = null;

	public const EXPORT_DIR_NAME = 'exportproducts';
	protected string      $name;
	public ?string        $baseUrl        = null;
	protected ?Site       $site           = null;
	protected ?SiteDomain $domain         = null;
	protected string      $currency       = 'CZK';
	protected ?Currency   $currencyEntity = null;

	/** @var Container|null @inject */
	public ?Container $container = null;

	/** @var object|null */
	public $orderGifts = null;

	/** @var PaymentSpeditions|null @inject */
	public ?PaymentSpeditions $paymentSpeditions = null;

	/** @var EntityManagerDecorator|null @inject */
	public ?EntityManagerDecorator $em = null;

	/** @var Translator|null @inject */
	public ?Translator $translator = null;

	/** @var Settings|null @inject */
	public ?Settings $settings;

	/** @var EventDispatcher|null @inject */
	public ?EventDispatcher $eventDispatcher;

	/** @var object|TypesList|null */
	public ?object $measuringTypesList = null;

	protected ?array $cSpeditions              = null;
	protected ?array $cFeatureExport           = null;
	protected ?array $cFeatureNames            = null;
	protected ?array $cFeatureValues           = null;
	protected ?array $cGoogleFeatureTypes      = null;
	protected ?array $cGoogleFeatureValueTypes = null;
	protected ?array $cZboziFeatureTypes       = null;
	protected ?array $cZboziFeatureUnit        = null;
	protected ?array $cZboziFeatureValueUnit   = null;
	protected ?array $cTags                    = null;
	protected array  $usedProducts             = [];

	public function getFilePathTemp(): string
	{
		return TMP_DIR . sprintf("/" . self::EXPORT_DIR_NAME . "/%s-%s-%s.xml",
				$this->getName(),
				$this->site->getIdent(),
				$this->domain->getLang(),
			);
	}

	public function getStockFilePathTemp(): string
	{
		return TMP_DIR . sprintf("/" . self::EXPORT_DIR_NAME . "/%s-%s-%s-stock.xml",
				$this->getName(),
				$this->site->getIdent(),
				$this->domain->getLang(),
			);
	}

	public function getFilePath(): string
	{
		return sprintf(WWW_DIR . '/' . self::EXPORT_DIR_NAME . '/%s-%s-%s.xml',
			$this->getName(),
			$this->site->getIdent(),
			$this->domain->getLang(),
		);
	}

	public function getStockFilePath(): string { return str_replace('.xml', '-stock.xml', $this->getFilePath()); }

	/**
	 * @return OrderGifts|object|null
	 */
	public function getOrdersGifs()
	{
		if (!$this->orderGifts && $this->container->hasService('eshopGifts.front.orderGifts')) {
			$this->orderGifts = $this->container->getService('eshopGifts.front.orderGifts');
		}

		return $this->orderGifts;
	}

	public function startExport(): void
	{
		if ($this->container->hasService('measuringCodes.front.typesList')) {
			$this->measuringTypesList = $this->container->getService('measuringCodes.front.typesList');
		}

		FileSystem::createDir(dirname($this->getFilePathTemp()));
		$this->fileHandle = fopen($this->getFilePathTemp(), 'w');
	}

	public function startExportStock(): void
	{
		FileSystem::createDir(dirname($this->getStockFilePathTemp()));
		$this->fileHandleStock = fopen($this->getStockFilePathTemp(), 'w');
	}

	public function writeToFile(string $data): void
	{
		if ($this->fileHandle) {
			fwrite($this->fileHandle, $data);
		}
	}

	public function writeToFileStock(string $data): void
	{
		if ($this->fileHandleStock) {
			fwrite($this->fileHandleStock, $data);
		}
	}

	/**
	 * @return void
	 */
	public function setBaseUrl(string $baseUrl) { $this->baseUrl = $baseUrl; }

	public function setSite(Site $site): void { $this->site = $site; }

	public function setDomain(SiteDomain $domain): void { $this->domain = $domain; }

	public function getName(): string { return $this->name; }

	public function setCurrency(string $currency): void { $this->currency = $currency; }

	public function setCurrencyEntity(Currency $currencyEntity): void { $this->currencyEntity = $currencyEntity; }

	public function getCurrencyEntity(): ?Currency { return $this->currencyEntity; }

	public function getCurrency(): string { return $this->currency; }

	/**
	 * @param Product $product
	 *
	 * @return array{
	 *     lang: string,
	 *     service: string,
	 *     status: int,
	 *     product: ?string,
	 *     productName: ?string,
	 *     productDescription: ?string,
	 *     categoryText: ?string,
	 *     bidCpc: ?int,
	 *     customLabel0: ?string,
	 *     customLabel1: ?string,
	 *     customLabel2: ?string,
	 *     customLabel3: ?string,
	 *     customLabel4: ?string,
	 *     firstTag: ?string,
	 * }
	 */
	public function prepareExportData(Product $product): array
	{
		$ef       = $product->getExtraFields();
		$export   = $ef['export'][$this->name];
		$category = [];

		if (isset($ef['categoryExports'][$this->name])) {
			$vals = $ef['categoryExports'][$this->name];

			$category['status']       = $vals['status'];
			$category['categoryText'] = $vals['categoryText'];

			if ($this->name === ExportEnums::SERVICE_FACEBOOK) {
				$category['categoryText'] = explode('|', (string) $category['categoryText'])[0];
			}

			for ($i = 0; $i <= 4; $i++) {
				$category['customLabel' . $i] = $vals['customLabel' . $i];
			}
		}

		if (!in_array($this->name, [ExportEnums::SERVICE_GOOGLE, ExportEnums::SERVICE_FACEBOOK], true)) {
			$tmp                   = $export;
			$export['product']     = $tmp['productName'];
			$export['productName'] = $tmp['product'];
			$tmp                   = null;
		}

		if (!$export) {
			$export = ['status' => 2];
		}

		if ($export['status'] === 2) {
			$export['status'] = $category['status'];
		}

		if (!$export['categoryText']) {
			$export['categoryText'] = $category['categoryText'];
		}

		if (!$export['productName']) {
			$export['productName'] = $product->getName();

			if (Config::load('exportGrader.addManufacturerToName') && $product->getManufacturer()) {
				$manufacturer = $product->getManufacturer()->name;

				if (!Strings::startsWith($export['productName'], $manufacturer)) {
					$export['productName'] = trim($manufacturer . ' ' . $export['productName']);
				}
			}
		}

		for ($i = 0; $i <= 4; $i++) {
			if ($category['customLabel' . $i]) {
				$export['customLabel' . $i] = $category['customLabel' . $i];
			}
		}

		if (!$export['product']) {
			$export['product'] = $export['productName'];
		}

		foreach (['product', 'productName'] as $v) {
			$export[$v] = preg_replace('/\s\s+/', ' ', $export[$v]);
		}

		if ($this->name === 'zbozi') {
			$export['firstTag'] = $product->tags ? $this->getFirstTag(array_keys($product->tags)) : null;
		}

		return $export;
	}

	/**
	 * @return void
	 */
	public function endExport()
	{
		fclose($this->fileHandle);
		FileSystem::rename($this->getFilePathTemp(), $this->getFilePath());
		@unlink($this->getFilePathTemp());
	}

	/**
	 * @return void
	 */
	public function endExportStock()
	{
		fclose($this->fileHandleStock);
		FileSystem::rename($this->getStockFilePathTemp(), $this->getStockFilePath());
		@unlink($this->getStockFilePathTemp());
	}

	public function getAllSpeditions(): array
	{
		$entityValue = '';
		$groupKey    = '';
		$country     = 'cz';
		if ($this->name === 'heureka') {
			$entityValue = 'heurekaId';
			$groupKey    = 'heurekaId';
		} else if ($this->name === 'zbozi') {
			$entityValue = 'zboziId';
			$groupKey    = 'heurekaId';
		} else if ($this->name === 'idealo') {
			$country     = 'de';
			$entityValue = 'idealoId';
			$groupKey    = 'idealoId';
		} else if ($this->name === 'google') {
			$country  = null;
			$groupKey = 'google';
		} else if ($this->name === 'facebook') {
			$groupKey = 'facebook';
		}

		if ($this->cSpeditions[$groupKey] === null) {
			$data = [];

			foreach ($this->paymentSpeditions->getAllPublishedByCountry($country) as $row) {
				$s = $row->getSpedition();
				if ($entityValue && !$s->{$entityValue} || (in_array($this->name, [ExportEnums::SERVICE_GOOGLE, ExportEnums::SERVICE_FACEBOOK], true) && $s->googleId == '0')) {
					continue;
				}

				$p = $row->getPayment();

				if (in_array($this->name, [ExportEnums::SERVICE_GOOGLE, ExportEnums::SERVICE_FACEBOOK], true) && $p->disableInGoogle) {
					continue;
				}

				$key = $s->getName() . ' - ' . $p->getName();
				if ($entityValue) {
					$key = $s->{$entityValue};
				}

				foreach ($row->getCountries() as $country) {
					/** @var Country $country */
					$data[$country->getId()][$key][] = [
						'spedition' => [
							'id'             => $s->getId(),
							'name'           => $s->getName(),
							'ident'          => $s->getIdent(),
							'price'          => $s->getPrice(),
							'from'           => $s->getAvailableFrom(),
							'to'             => $s->getAvailableTo(),
							'freeFrom'       => $s->getFreeFrom(),
							'isPickUp'       => $s->isPickup,
							'allowOverSized' => $s->allowOversized,
						],
						'payment'   => [
							'id'       => $p->getId(),
							'name'     => $p->getName(),
							'ident'    => $p->getIdent(),
							'price'    => $p->getPrice(),
							'from'     => $p->getAvailableFrom(),
							'to'       => $p->getAvailableTo(),
							'freeFrom' => $p->getFreeFrom(),
						],
					];
				}
			}

			$this->cSpeditions[$groupKey] = $data;
		}

		return $this->cSpeditions[$groupKey];
	}

	public function getSpeditionsForPrice(
		float $prodPrice,
		bool  $isOversize = false,
		bool  $disablePickUp = false,
		array $disabledSpedition = [],
		array $disabledPayments = []
	): array
	{
		$result = [];

		foreach ($this->getAllSpeditions() as $country => $speditions) {
			foreach ($speditions as $spedId => $rows) {
				$codAvailable = false;
				$skip         = true;
				$d            = [
					'price' => 999999,
					'cod'   => 999999,
				];

				foreach ($rows as $row) {
					$s = $row['spedition'];
					$p = $row['payment'];

					$d['speditionName'] = $s['name'];
					$d['paymentName']   = $p['name'];

					if (in_array($s['id'], $disabledSpedition) || in_array($p['id'], $disabledPayments)) {
						continue;
					}

					if ($disablePickUp && $s['isPickup'] === true
						|| $isOversize && $s['allowOverSized'] === false) {
						continue;
					}

					if ($prodPrice < $s['from'] || $prodPrice > $s['to']
						|| $prodPrice < $p['from'] || $prodPrice > $p['to']) {
						continue;
					}

					$pPrice = $p['freeFrom'] !== null && $prodPrice > $p['freeFrom'] ? 0 : $p['price'];
					$sPrice = $s['freeFrom'] !== null && $prodPrice > $s['freeFrom'] ? 0 : $s['price'];

					$price = $sPrice;

					if (strtolower($p['ident']) === 'cod') {
						$cod          = $price + $pPrice;
						$codAvailable = true;

						if ($d['cod'] > $cod) {
							$d['cod'] = $cod;
						}
					} else {
						$price += $pPrice;
					}

					if ($d['price'] > $price) {
						$d['price'] = $price;
					}

					$skip = false;
				}

				if ($skip) {
					continue;
				}

				if (!$codAvailable) {
					unset($d['cod']);
				}

				$result[$country][$spedId] = $d;
			}

			if (array_key_exists($country, $result)) {
				uasort($result[$country], static fn($a, $b) => $a['price'] <=> $b['price']);
			}
		}

		return $result;
	}

	/**
	 * @return false|mixed
	 */
	public function canExportFeature(int $featureId)
	{
		if ($this->cFeatureExport === null) {
			$this->cFeatureExport = [];

			foreach ($this->em->getRepository(Feature::class)->createQueryBuilder('f')
				         ->select('f.id, f.exportHeureka as heureka, f.exportZbozi as zbozi, f.zboziType, f.zboziUnit, f.exportGoogle as google, f.googleType, f.exportIdealo as idealo')
				         ->getQuery()->getArrayResult() as $row) {
				$this->cFeatureExport[$row['id']] = $row;
				if ($row['googleType']) {
					$this->cGoogleFeatureTypes[$row['id']] = $row['googleType'];
				}

				if ($row['zboziType']) {
					$this->cZboziFeatureTypes[$row['id']] = $row['zboziType'];
				}

				if ($row['zboziUnit']) {
					$this->cZboziFeatureUnit[$row['id']] = $row['zboziUnit'];
				}
			}
		}

		return $this->cFeatureExport[$featureId][$this->name] ?? false;
	}

	/**
	 * @return mixed|null
	 */
	public function getExportFeatureName(int $featureId)
	{
		if ($this->cFeatureNames === null) {
			$this->cFeatureNames = [];

			foreach ($this->em->getRepository(FeatureTexts::class)->createQueryBuilder('ft')
				         ->select('IDENTITY(ft.id) as id, ft.lang, ft.heurekaName as heureka, ft.zboziName as zbozi, ft.googleName as google, ft.idealoName as idealo')
				         ->getQuery()->getArrayResult() as $row) {
				$this->cFeatureNames[$row['lang']][$row['id']] = $row;
			}
		}

		return $this->cFeatureNames[$this->domain->getLang()][$featureId][$this->name] ?? null;
	}

	/**
	 * @return mixed|null
	 */
	public function getExportFeatureValue(int $featureValueId)
	{
		if ($this->cFeatureValues === null) {
			$this->cFeatureValues         = [];
			$this->cZboziFeatureValueUnit = [];

			foreach ($this->em->getRepository(FeatureValueTexts::class)->createQueryBuilder('fvt')
				         ->select('IDENTITY(fvt.id) as id, fvt.lang, fvt.heurekaName as heureka, fvt.zboziName as zbozi, fvt.googleName as google, fvt.idealoName as idealo')
				         ->getQuery()->getArrayResult() as $row) {
				$this->cFeatureValues[$row['lang']][$row['id']] = $row;
			}

			foreach ($this->em->getConnection()->executeQuery("SELECT id, zbozi_unit FROM eshop_catalog__feature_value WHERE zbozi_unit IS NOT NULL")
				         ->iterateAssociative() as $row) {
				/** @var array $row */
				if ($row['zbozi_unit']) {
					$this->cZboziFeatureValueUnit[$row['id']] = $row['zbozi_unit'];
				}
			}
		}

		return $this->cFeatureValues[$this->domain->getLang()][$featureValueId][$this->name] ?? null;
	}

	/**
	 * @return mixed|null
	 */
	public function getZboziFeatureType(int $featureId)
	{
		if ($this->cZboziFeatureTypes === null) {
			$this->cZboziFeatureTypes = [];

			$this->canExportFeature($featureId);
		}

		return $this->cZboziFeatureTypes[$featureId] ?? null;
	}

	/**
	 * @return mixed|null
	 */
	public function getZboziFeatureUnit(int $featureId, ?int $valueId = null)
	{
		if ($this->cZboziFeatureUnit === null) {
			$this->cZboziFeatureUnit = [];

			$this->canExportFeature($featureId);
		}

		if ($this->cZboziFeatureValueUnit === null && $valueId) {
			$this->getExportFeatureValue($valueId);
		}

		if (isset($this->cZboziFeatureValueUnit[$valueId])) {
			return $this->cZboziFeatureValueUnit[$valueId];
		}

		return $this->cZboziFeatureUnit[$featureId] ?? null;
	}

	/**
	 * @return mixed|null
	 */
	public function getGoogleFeatureType(int $featureId)
	{
		if ($this->cGoogleFeatureTypes === null) {
			$this->cGoogleFeatureTypes = [];

			$this->canExportFeature($featureId);
		}

		return $this->cGoogleFeatureTypes[$featureId] ?? null;
	}

	/**
	 * @return mixed|null
	 */
	public function getGoogleFeatureValueType(int $featureValueId)
	{
		if ($this->cGoogleFeatureValueTypes === null) {
			$this->cGoogleFeatureValueTypes = [];

			foreach ($this->em->getRepository(FeatureValue::class)->createQueryBuilder('fv')
				         ->select('fv.id, fv.googleType')
				         ->where('fv.googleType IS NOT NULL')
				         ->getQuery()->getArrayResult() as $row) {
				if ($row['googleType']) {
					$tmp                                        = explode('__', $row['googleType']);
					$this->cGoogleFeatureValueTypes[$row['id']] = $tmp[1] ?? $tmp[0];
				}
			}
		}

		return $this->cGoogleFeatureValueTypes[$featureValueId] ?? null;
	}

	public function getFirstTag(array $tags): ?string
	{
		if (!$tags) {
			return null;
		}

		if ($this->cTags === null) {
			$this->cTags = [];
			foreach ($this->em->getRepository(Tag::class)->createQueryBuilder('t')
				         ->select('t.id, t.type, tt.name')
				         ->innerJoin('t.texts', 'tt', Join::WITH, 'tt.lang = :lang')
				         ->setParameters([
					         'lang' => $this->domain->getLang(),
				         ])
				         ->where('t.ppcPriority IS NOT NULL')
				         ->orderBy('t.ppcPriority', 'ASC')
				         ->getQuery()->getArrayResult() as $row) {
				$this->cTags[$row['name']] = $row['name'];
			}
		}

		return key(array_intersect($this->cTags, $tags)) ?? null;
	}

	protected function parseImgUrl(string $url): string
	{
		$url = str_replace(' ', '%20', $url);

		return Strings::startsWith($url, 'http') ? $url : $this->baseUrl . $url;
	}
}
