<?php declare(strict_types = 1);

namespace EshopProductsComparison\FrontModule\Model\Export;

use Currency\Model\Currencies;
use DOMDocument;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Helpers\ProductConditionEnum;
use EshopCatalog\Model\Helpers\ZboziExportEnums;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopProductsComparison\FrontModule\Model\Event\XmlShopItemEvent;
use EshopProductsComparison\FrontModule\Model\Provider\IXmlExportService;
use EshopProductsComparison\Model\EshopProductsComparisonConfig;
use EshopSales\FrontModule\Model\Dao\OrderSale;
use EshopSales\FrontModule\Model\OrderSales;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;

class ZboziXmlService extends XmlService implements IXmlExportService
{
	protected string $name = 'zbozi';

	protected ?bool $extendedReturn = null;

	public function startExport(): void
	{
		parent::startExport();

		$this->writeToFile('<?xml version="1.0" encoding="utf-8"?>' . PHP_EOL .
			'<SHOP xmlns="http://www.zbozi.cz/ns/offer/1.0">' . PHP_EOL);
	}

	/**
	 * @return void
	 */
	public function writeItem(Product $product)
	{
		if (isset($this->usedProducts[$product->getId()])) {
			return;
		}

		$currencyEntity = $this->getCurrencyEntity();
		if (!$currencyEntity) {
			return;
		}

		$this->usedProducts[$product->getId()] = true;

		$export = $this->prepareExportData($product);

		if (
			$export['status'] === 0
			|| array_key_exists($this->name, $product->getExtraField('disabledByManufacturer') ?? [])
		) {
			return;
		}

		$depots = [];
		foreach (EshopProductsComparisonConfig::load('zbozi.shopDepots') as $k => $v) {
			if (is_array($v)) {
				if ($k === $this->site->getIdent()) {
					foreach ($v as $v2) {
						$depots[] = $v2;
					}
				}
			} else {
				$depots[] = $v;
			}
		}

		if (empty($depots) && $this->measuringTypesList->getType('zboziConversion')) {
			$zboziType = $this->measuringTypesList->getType('zboziConversion');

			if ($zboziType && $zboziType->getField('id_firmy_cz')) {
				$val = $zboziType->getField('id_firmy_cz')->getValue();

				if ($val) {
					$depots[] = $val;
				}
			}
		}

		$dom               = new DOMDocument;
		$dom->formatOutput = true;
		$dom->encoding     = 'utf-8';
		$shopitem          = $dom->createElement('SHOPITEM');

		$child = $dom->createElement('ITEM_ID', (string) $product->getId());
		$shopitem->appendChild($child);

		if ($product->variantId) {
			$child = $dom->createElement('ITEMGROUP_ID', (string) $product->variantId);
			$shopitem->appendChild($child);
		}

		$child     = $dom->createElement('PRODUCTNAME');
		$childText = $dom->createTextNode((string) $export['productName']);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child     = $dom->createElement('PRODUCT');
		$childText = $dom->createTextNode((string) $export['product']);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child     = $dom->createElement('DESCRIPTION');
		$childText = $dom->createCDATASection(Strings::normalize(strip_tags(html_entity_decode((string) ($export['productDescription'] ?: $product->getDescription())))));
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child     = $dom->createElement('URL');
		$childText = $dom->createCDATASection((string) $product->link);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		if ($product->getRetailPrice() > $product->getPrice()) {
			$child = $dom->createElement('PRICE_VAT', number_format(round($product->getPrice(), $currencyEntity->decimals), 2, ".", ""));
			$shopitem->appendChild($child);
			$child = $dom->createElement('PRICE_BEFORE_DISCOUNT', number_format(round($product->getRetailPrice(), $currencyEntity->decimals), 2, ".", ""));
			$shopitem->appendChild($child);
		} else {
			$child = $dom->createElement('PRICE_VAT', number_format(round($product->getPrice(), $currencyEntity->decimals), 2, ".", ""));
			$shopitem->appendChild($child);
		}

		$child = $dom->createElement('DELIVERY_DATE', (string) $product->getAvailability()->getDelay());
		$shopitem->appendChild($child);

		foreach ($this->getSpeditionsForPrice($product->getPrice(), $product->isOversize, $product->hasDisabledPickupPoints(), $product->disabledSpeditions, $product->disabledPayments) as $country => $speditions) {
			if (Strings::upper($country) !== Strings::upper($this->domain->getDefaultCountry())) {
				continue;
			}

			foreach ($speditions as $spedName => $spedition) {
				$delivery = $dom->createElement('DELIVERY');

				$child = $dom->createElement('DELIVERY_ID', (string) $spedName);
				$delivery->appendChild($child);
				$child = $dom->createElement('DELIVERY_PRICE', number_format(round($spedition['price'], $currencyEntity->decimals), 2, '.', ''));
				$delivery->appendChild($child);

				if ($spedition['cod']) {
					$child = $dom->createElement('DELIVERY_PRICE_COD', (string) $spedition['cod']);
					$delivery->appendChild($child);
				}

				$shopitem->appendChild($delivery);
			}
		}

		if ($product->getGallery()) {
			$coverId = null;
			if ($product->getGallery()->getCover()) {
				$coverId = $product->getGallery()->getCover()->id;
				$child   = $dom->createElement('IMGURL', $this->parseImgUrl($product->getGallery()->getCover()->getFilePath()));
				$shopitem->appendChild($child);
			}

			foreach ($product->getGallery()->getImages() as $img) {
				if ($img->id == $coverId) {
					continue;
				}

				$child = $dom->createElement('IMGURL_ALTERNATIVE', $this->parseImgUrl($img->getFilePath()));
				$shopitem->appendChild($child);
			}
		}

		if ($product->getEan()) {
			$child = $dom->createElement('EAN', $product->getEan());
			$shopitem->appendChild($child);
		}

		if ($export['categoryText']) {
			$child = $dom->createElement('CATEGORYTEXT', (string) $export['categoryText']);
			$shopitem->appendChild($child);
		}

		if ($product->condition !== ProductConditionEnum::new) {
			$child = $dom->createElement('CONDITION', ProductConditionEnum::toSeznam($product->condition));
			$shopitem->appendChild($child);

			if ($product->conditionDescription && Config::load('product.allowConditionDesc')) {
				$child     = $dom->createElement('CONDITION_DESC');
				$childText = $dom->createCDATASection(Strings::normalize(strip_tags($product->conditionDescription)));
				$child->appendChild($childText);
				$shopitem->appendChild($child);
			}
		}

		if ($product->defaultCategory) {
			$child = $dom->createElement('CUSTOM_LABEL_0', (string) $product->defaultCategory->name);
			$shopitem->appendChild($child);
		}

		if ($export['customLabel1']) {
			$param = $dom->createElement('CUSTOM_LABEL_1', $export['customLabel1']);
			$shopitem->appendChild($param);
		}

		if ($export['firstTag']) {
			$child = $dom->createElement('CUSTOM_LABEL_3', $export['firstTag']);
			$shopitem->appendChild($child);
		}

		if ($product->getManufacturer() && trim($product->getManufacturer()->name)) {
			$child = $dom->createElement('MANUFACTURER', (string) $product->getManufacturer()->name);
			$shopitem->appendChild($child);
		}

		$usedFeatures = [];
		foreach ($product->getFeatures() as $feature) {
			if (
				!$this->canExportFeature((int) $feature->idFeature)
				|| isset($usedFeatures[$feature->idFeature])
				|| !$feature->showInExport
			) {
				continue;
			}

			$zboziValue = $this->getExportFeatureValue((int) $feature->idFeatureValue) ?: $feature->value;
			$zboziType  = $this->getZboziFeatureType((int) $feature->idFeature);
			$zboziUnit  = $this->getZboziFeatureUnit((int) $feature->idFeature, (int) $feature->idFeatureValue);

			if ($zboziType) {
				if ($zboziType === ZboziExportEnums::WARRANTY) {
					$child = $dom->createElement('WARRANTY', (string) $feature->rawValue);
					$shopitem->appendChild($child);
				}
			} else {
				$param = $dom->createElement('PARAM');

				$child = $dom->createElement('PARAM_NAME', $this->getExportFeatureName((int) $feature->idFeature) ?: $feature->name);
				$param->appendChild($child);
				$child = $dom->createElement('VAL', htmlspecialchars($zboziValue, ENT_XML1 | ENT_QUOTES, 'UTF-8'));
				$param->appendChild($child);

				if ($zboziUnit) {
					$child = $dom->createElement('UNIT', htmlspecialchars($zboziUnit, ENT_XML1 | ENT_QUOTES, 'UTF-8'));
					$param->appendChild($child);
				}

				$shopitem->appendChild($param);
			}

			$usedFeatures[$feature->idFeature] = $feature->idFeatureValue;
		}

		if ($export['bidCpc']) {
			$child = $dom->createElement('MAX_CPC', (string) $export['bidCpc']);
			$shopitem->appendChild($child);
		}

		if (
			!empty($product->tags)
			&& class_exists('\EshopSales\FrontModule\Model\OrderSales')
			&& class_exists('\Currency\Model\Currencies')
			&& $this->container->hasService('eshopSales.front.orderSales')
		) {
			/** @var OrderSales $salesService */
			$salesService = $this->container->getService('eshopSales.front.orderSales');

			/** @var Currencies $currencies */
			$currencies = $this->container->getService('currency.currencies');

			$salesService->loadProductTagSalePrice($product);
			if ($product->moreData['tagSalesPrice']) {
				/** @var OrderSale $dto */
				$dto = $product->moreData['tagSalesPrice']['dto'];

				$sale = '';
				if ($dto->type === OrderDiscount::TYPE_PERCENT) {
					$sale = $dto->value . ' %';
				} else if ($dto->type === OrderDiscount::TYPE_FIX) {
					$sale = $dto->value . ' ' . $currencies->getCurrent()->symbol;
				}
				$codeValue = $dto->code ?: $dto->getAutoSaleCode();

				// sales voucher
				$dateFrom = ($dto->dateFrom ?: (new DateTime())->modify('first day of this month'));
				$dateTo   = ($dto->dateTo ?: (new DateTime())->modify('last day of this month'));

				$dateFromStr = $dateFrom->format('Y-m-d') . 'T00:00:00';
				$dateToStr   = $dateTo->format('Y-m-d') . 'T23:59:59';

				$child = $dom->createElement('SALES_VOUCHER');
				$child->appendChild($dom->createElement('CODE', $codeValue));
				$child->appendChild($dom->createElement('PRICE_WITH_VOUCHER', (string) $product->moreData['tagSalesPrice']['price']));
				$child->appendChild($dom->createElement('DATE_FROM', $dateFromStr));
				$child->appendChild($dom->createElement('DATE_TO', $dateToStr));
				$child->appendChild($dom->createElement('TEXT', Strings::truncate($this->translator->translate('eshopProductsComparisonFront.exportSaleText', [
					'sale' => $sale,
					'code' => $codeValue,
				]), 128)));

				$shopitem->appendChild($child);
			}
		}

		$hasGift = false;
		foreach ($product->getGifts() as $gift) {
			/** @phpstan-ignore-next-line */
			$child = $dom->createElement('EXTRA_MESSAGE', 'free_gift');
			$shopitem->appendChild($child);

			/** @phpstan-ignore-next-line */
			$child = $dom->createElement('FREE_GIFT_TEXT', $gift->getName());
			$shopitem->appendChild($child);

			$hasGift = true;
			break;
		}

		if (!$hasGift && $this->getOrdersGifs()) {
			$giftsCount = count($this->getOrdersGifs()->findGifts($product->getPrice()));

			if ($giftsCount > 0) {
				/** @phpstan-ignore-next-line */
				$child = $dom->createElement('EXTRA_MESSAGE', 'free_gift');
				$shopitem->appendChild($child);

				/** @phpstan-ignore-next-line */
				$child = $dom->createElement('FREE_GIFT_TEXT', $this->translator->translate('eshopProductsComparisonFront.giftsText'));
				$shopitem->appendChild($child);
			}
		}

		if ($product->getQuantity() > 0) {
			foreach ($depots as $depot) {
				$child = $dom->createElement('SHOP_DEPOTS', (string) $depot);
				$shopitem->appendChild($child);
			}
		}

		if ($this->canAddExtendedReturn()) {
			$child = $dom->createElement('EXTRA_MESSAGE', 'extended_return');
			$shopitem->appendChild($child);
		}

		$event = new XmlShopItemEvent($dom, $shopitem, $product, 'zbozi');
		$this->eventDispatcher->dispatch($event, 'eshopProductsComparison.xmlShopItem');

		$dom->appendChild($shopitem);

		$this->writeToFile($dom->saveXML($dom->documentElement) . PHP_EOL);
	}

	/**
	 * @return void
	 */
	public function endExport()
	{
		$this->writeToFile('</SHOP>');

		parent::endExport();
	}

	protected function canAddExtendedReturn(): bool
	{
		if ($this->extendedReturn === null) {
			$this->extendedReturn = false;

			$zboziSettingsKey = 'eshopCatalog' . Strings::firstUpper($this->site->getIdent()) . 'XmlFeedZbozi';
			$extendedReturn   = $this->settings->get($zboziSettingsKey . 'ExtendedReturn');

			if ($extendedReturn) {
				$extendedReturnDateTo = $this->settings->get($zboziSettingsKey . 'ExtendedReturnDateTo');

				$toDate = $extendedReturnDateTo ? DateTime::createFromFormat('d.m.Y H:i', $extendedReturnDateTo) : null;
				$now    = (new DateTime())
					->setTime(0, 0, 0)
					->format('Y-m-d H:i');

				if (!$toDate || $now <= $toDate->format('Y-m-d H:i')) {
					$this->extendedReturn = true;
				}
			}
		}

		return $this->extendedReturn;
	}
}
