<?php declare(strict_types = 1);

namespace EshopProductsComparison\FrontModule\Model;

use Core\Model\Dao\Site;
use Core\Model\Dao\SiteDomain;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseService;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Currency\Model\Currencies;
use EshopProductsComparison\FrontModule\Model\Export\Data;
use EshopProductsComparison\FrontModule\Model\Export\XmlExportCollection;
use EshopProductsComparison\FrontModule\Model\Export\XmlService;
use EshopProductsComparison\Model\EshopProductsComparisonConfig;
use Nette\Http\Request;
use Tracy\Debugger;

class ExportFacade extends BaseService
{
	protected EntityManagerDecorator $em;
	protected XmlExportCollection    $xmlExport;
	protected Request                $httpRequest;
	protected Data                   $dataService;
	protected EventDispatcher        $eventDispatcher;
	protected Currencies             $currencies;

	public ?string $currency     = null;
	public array   $writtenLinks = [];

	public function __construct(
		EntityManagerDecorator $em,
		XmlExportCollection    $xmlExport,
		Request                $request,
		Data                   $data,
		EventDispatcher        $eventDispatcher,
		Currencies             $currencies
	)
	{
		$this->em              = $em;
		$this->xmlExport       = $xmlExport;
		$this->httpRequest     = $request;
		$this->dataService     = $data;
		$this->eventDispatcher = $eventDispatcher;
		$this->currencies      = $currencies;
	}

	public function generateExport(array $types, Site $site, SiteDomain $domain): void
	{
		$this->em->getConfiguration()->setSQLLogger();

		if (php_sapi_name() !== 'cli') {
			set_time_limit(6000);
		}

		if ($this->currency) {
			$this->dataService->currency = $this->currency;
		}

		$currencyEntity = $this->currencies->getCurrent();
		if (!$currencyEntity) {
			throw new \Exception('Currency not found');
		}

		// Načtení vybraných exportů
		/** @var XmlService[] $services */
		$services = [];
		/** @var XmlService $v */
		foreach ($this->xmlExport->getServices() as $k => $v) {
			if (in_array($k, $types)) {
				$services[$k] = $v;
			}
		}

		// Inicializace exportu
		$baseUrl = 'https://' . $domain->getDomain();

		foreach ($services as $service) {
			$service->setSite($site);
			$service->setDomain($domain);
			$service->setBaseUrl($baseUrl);
			$service->setCurrency($this->currency);
			$service->setCurrencyEntity($currencyEntity);
			$service->startExport();
		}

		$i         = 1;
		$start     = 0;
		$limit     = (int) EshopProductsComparisonConfig::load('productsLoopCount', 5000);
		$loop      = 1;
		$loopLimit = 100;

		$products = $this->dataService->getProductsData($start, $limit);

		do {
			$nextLoop = $products['baseCount'] >= $limit && $loop <= $loopLimit;

			foreach ($products['products'] as $product) {
				if (!isset($this->writtenLinks[$product->link])) {
					foreach ($services as $service) {
						$service->writeItem($product);
					}
				}

				$this->writtenLinks[$product->link][] = $product->getId();
			}

			$start    = $limit * $i++;
			$products = $this->dataService->getProductsData($start, $limit);

			$loop++;

			$this->em->clear();
			gc_collect_cycles();
		} while ($nextLoop);

		foreach ($services as $service) {
			$service->endExport();
		}

		$event = new Event();
		$this->eventDispatcher->dispatch($event, 'eshopProductsComparison.exportFinished');

		// Duplicitni url ve feedu
		/** @phpstan-ignore-next-line */
		if (false) {
			$linksToEmail = [];
			foreach ($this->writtenLinks as $link => $ids) {
				if (count($ids) > 1) {
					$linksToEmail[$link] = $ids;
				}
			}

			if (!empty($linksToEmail)) {
				$str = '';
				foreach ($linksToEmail as $link => $ids) {
					$str .= $link . ' - ' . implode(', ', $ids) . "<br>";
				}
				Debugger::log($domain->getDomain() . ' - ' . $domain->getLang(), 'EshopProductsExport');
				Debugger::log($linksToEmail, 'EshopProductsExport');
				LogNotifier::toDevelopers($str, 'Duplicitní odkazy ve feedech - ' . $domain->getDomain() . ' - ' . $domain->getLang() . ', ' . $this->currency);
			}
		}
	}
}
