<?php declare(strict_types = 1);

namespace EshopProductsComparison\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Components\Products\ProductForm;
use EshopCatalog\Model\Entities\Product;
use EshopProductsComparison\Model\Entities\ProductExport;
use EshopProductsComparison\Model\Helpers\ExportEnums;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class ProductFormSubscriber implements EventSubscriberInterface
{
	public function __construct(
		protected EntityManagerDecorator $em,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			ProductForm::class . '::createForm'  => 'createForm',
			ProductForm::class . '::formSuccess' => 'formSuccess',
			ProductForm::class . '::setProduct'  => 'setProduct',
		];
	}

	public function createForm(CreateFormEvent $event): void
	{
		$baseContainer = $event->form->addContainer('productsComparison');
		$baseContainer->addCustomData('template', __DIR__ . '/ProductForm.latte');

		foreach (ExportEnums::$services as $serviceName => $v) {
			$container = new BaseContainer;

			$container->addBool('status', 'eshopProductsComparison.comparisons.status', ExportEnums::$statuses)
				->setDefaultValue(2)
				->setIsMultilanguage();

			foreach ($v['fields'] as $k => $f) {
				$labelBase = 'eshopProductsComparison.comparisons.' . $serviceName . '.';

				if (is_array($f)) {
					if ($k === 'textarea') {
						foreach ($f as $f2) {
							if (!is_string($f2)) {
								continue;
							}

							$container->addTextArea($f2, $labelBase . $f2)
								->setHtmlAttribute('rows', 6)
								->setIsMultilanguage();
						}
					}
				} else {
					$container->addText($f, $labelBase . $f)
						->setMaxLength(255)
						->setIsMultilanguage();
				}
			}

			$container->getComponent('categoryText')
				->setHtmlAttribute('data-autocomplete-url', 'https://api-evidence.pshk.cz/v1/cms/' . $serviceName . '-export-categories/[data-content-lang]');

			if ($serviceName === ExportEnums::SERVICE_FACEBOOK) {
				$container->getComponent('categoryText')
					->setHtmlAttribute('data-select-item-data', 'id,name')
					->setIsMultilanguage(false);
			}

			$baseContainer->addComponent($container, $serviceName);
		}
	}

	public function formSuccess(FormSuccessEvent $event): void
	{
		/** @var Product $product */
		$product = $event->custom['entity'];

		$langValues = [];
		foreach ($event->form->convertMultilangValuesToArray() as $l => $vals) {
			$langValues[$l] = $vals['productsComparison'];
		}

		$fbCategoryText = $event->form->getValues()->productsComparison->facebook->categoryText ?? null;

		$exports     = [];
		$exportsVals = [];
		foreach ((array) $this->em->getRepository(ProductExport::class)->createQueryBuilder('ppe')
			->where('ppe.id = :id')->setParameter('id', $product->getId())
			->getQuery()->getResult() as $row) {
			/** @var ProductExport $row */
			$exports[$row->service][$row->lang] = $row;
		}

		foreach (array_keys(ExportEnums::$services) as $serviceName) {
			foreach ($langValues as $lang => $vals) {
				if (isset($vals[$serviceName])) {
					$exportsVals[$lang][$serviceName] = $vals[$serviceName];
				}

				if ($serviceName === ExportEnums::SERVICE_FACEBOOK && $fbCategoryText) {
					$exportsVals[$lang][$serviceName]['categoryText'] = $fbCategoryText;
				}
			}
		}

		foreach ($exportsVals as $lang => $services) {
			foreach ($services as $serviceName => $vals) {
				if (isset($exports[$serviceName][$lang])) {
					$export = $exports[$serviceName][$lang];
				} else {
					$export = new ProductExport($product, $lang, $serviceName);
				}

				$export->status             = (int) $vals['status'];
				$export->product            = $vals['product'] ?: null;
				$export->productName        = $vals['productName'] ?: null;
				$export->categoryText       = $vals['categoryText'] ?: null;
				$export->bidCpc             = $vals['bidCpc'] ?: null;
				$export->productDescription = $vals['productDescription'] ?: null;

				foreach ([0, 1, 2, 3, 4] as $cfI) {
					$export->{'customLabel' . $cfI} = $vals['customLabel' . $cfI] ?: null;
				}

				$this->em->persist($export);
			}
		}
	}

	public function setProduct(SetFormDataEvent $event): void
	{
		/** @var Product $product */
		$product = $event->entity;
		$form    = $event->form->getComponent('productsComparison', false);

		if (!$form) {
			return;
		}

		$exports = [];
		foreach ($this->em->getRepository(ProductExport::class)->createQueryBuilder('ppe')
			         ->where('ppe.id = :id')->setParameter('id', $product->getId())
			         ->getQuery()->getArrayResult() as $row) {
			/** @var array $row */
			foreach ($row as $k => $v) {
				if (in_array($k, ['service', 'lang', 'id'])) {
					continue;
				}

				if (isset($form[$row['service']][$k])) {
					$exports[$row['service']][$k][$row['lang']] = $v;
				}
			}
		}

		$event->form->getComponent('productsComparison')->setDefaults($exports);
	}
}
