<?php declare(strict_types = 1);

namespace EshopProductsComparison\Console;

use Core\Model\Lang\Langs;
use Core\Model\Sites;
use Currency\Model\Currencies;
use Doctrine;
use EshopProductsComparison\FrontModule\Model\ExportFacade;
use EshopProductsComparison\Model\EshopProductsComparisonConfig;
use Exception;
use Nette\Localization\Translator;
use Override;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

#[AsCommand(name: 'eshopproductscomparison:generate-export')]
class GenerateExport extends Command
{
	public function __construct(
		public Sites        $sites,
		public ExportFacade $exportFacade,
		public Translator   $translator,
		public Langs        $langs,
	)
	{
		parent::__construct();
	}

	#[Override]
	protected function configure(): void
	{
		$this->setName('eshopproductscomparison:generate-export')
			->setDescription('')
			->addOption('types', 'y', InputOption::VALUE_REQUIRED)
			->addOption('site', 's', InputOption::VALUE_REQUIRED)
			->addOption('lang', 'l', InputOption::VALUE_REQUIRED)
			->addOption('currency', 'c', InputOption::VALUE_REQUIRED);
	}

	#[Override]
	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		ini_set('memory_limit', EshopProductsComparisonConfig::loadScalar('consoleMemoryLimit') ?: '2G');
		$output->writeln('GENERATE EXPORT');

		try {
			/** @var string $siteIdent */
			$siteIdent = $input->getOption('site');
			/** @var string $lang */
			$lang = $input->getOption('lang');
			/** @var string $currency */
			$currency = $input->getOption('currency');
			/** @var string $typesString */
			$typesString = $input->getOption('types');
			$types       = explode(',', (string) $typesString);

			Sites::$currentIdentOverride     = $siteIdent;
			Sites::$currentLangOverride      = $lang;
			Currencies::$currentCodeOverride = $currency;
			$this->exportFacade->currency    = $currency;
			/** @phpstan-ignore-next-line */
			$this->translator->setLocale($lang);
			$this->langs->setDefault($lang);

			$site = $this->sites->getSites()[$siteIdent] ?? null;
			if (!$site) {
				return Command::FAILURE;
			}

			$domain = $site->getDomains()[$lang] ?? null;
			if (!$domain) {
				$domain = $site->getCurrentDomain();

				if (!$domain) {
					$output->writeln('<error>[ERROR] - Domain not found for site: ' . $siteIdent . '</error>');

					return Command::FAILURE;
				}
			}

			$domain->setLang($lang);

			$this->exportFacade->generateExport($types, $site, $domain);

			$output->writeLn('<info>[OK] - generated</info>');

			return Command::SUCCESS;
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return Command::FAILURE;
	}
}
