<?php declare(strict_types = 1);

namespace EshopProductsComparison\FrontModule\Model\Export;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Contributte\Translation\Translator;
use Core\Model\Dao\SiteDomain;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use Currency\Model\Config;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\Model\PaymentSpeditions;
use EshopProductsComparison\Model\Entities\CategoryExport;
use EshopProductsComparison\Model\Entities\ManufacturerExport;
use EshopProductsComparison\Model\Entities\ProductExport;
use EshopProductsComparison\Model\EshopProductsComparisonConfig;

class Data
{
	protected ?array $cCategories            = null;
	public ?string   $currency               = null;
	protected ?array $cDisabledManufacturers = null;

	public function __construct(
		protected EntityManagerDecorator $em,
		protected Translator             $translator,
		protected Sites                  $sites,
		protected Categories             $categories,
		protected ProductsFacade         $productsFacade,
		protected PaymentSpeditions      $paymentSpeditions,
	)
	{
	}

	public function getProductsData(int $start = 0, int $limit = 1000): array
	{
		$allIds                = [];
		$exports               = [];
		$categoryExports       = [];
		$result                = [
			'products' => [],
		];
		$minimumPrice          = EshopProductsComparisonConfig::load('exportMinimumPrice', 1);
		$defaultDomainCurrency = null;

		$currentSite = $this->sites->getCurrentSite();
		if ($currentSite->getCurrentDomain() instanceof SiteDomain) {
			$defaultDomainCurrency = $currentSite->getCurrentDomain()->defaultCurrency;
		}

		foreach ((new ProductQuery($this->translator->getLocale()))
			         ->onlyInStockOrSupplier()
			         ->selectIds()
			         ->inSite($this->sites->getCurrentSite()->getIdent())
			         ->hasPrice()
			         ->getQueryBuilder($this->em->getRepository(Product::class))
			         ->setMaxResults($limit)->getQuery()->setFirstResult($start)->getScalarResult() as $row) {
			/** @var array $row */
			$allIds[] = $row['id'];
		}

		$result['baseCount'] = count($allIds);

		if ($this->cDisabledManufacturers === null) {
			$this->cDisabledManufacturers = [];

			foreach ($this->em->getRepository(ManufacturerExport::class)->createQueryBuilder('me')
				         ->select('IDENTITY(me.id) as id, me.service, me.allowExport, me.lang')
				         ->getQuery()->getArrayResult() as $row) {
				/** @var array $row */
				if ($row['allowExport'] === 0) {
					$this->cDisabledManufacturers[$row['id']][$row['service']] = true;
				}
			}
		}

		if ($allIds !== []) {
			foreach (array_chunk($allIds, 500) as $ids) {
				foreach ($this->em->getRepository(CategoryExport::class)->createQueryBuilder('ce')
					         ->where('ce.lang = :lang')->setParameter('lang', $this->translator->getLocale())
					         ->getQuery()->getArrayResult() as $row) {
					/** @var array $row */
					$categoryExports[$row['id']][$row['service']] = $row;
				}

				foreach ($this->em->getRepository(ProductExport::class)->createQueryBuilder('pe')
					         ->where('pe.id IN (' . implode(',', $ids) . ')')->andWhere('pe.lang = :lang')
					         ->setParameters(new ArrayCollection([new Parameter('lang', $this->translator->getLocale())]))
					         ->getQuery()->getArrayResult() as $row) {
					/** @var array $row */
					$exports[$row['id']][$row['service']] = $row;
				}

				foreach ($this->productsFacade->getProducts($ids) as $product) {
					if (
						!$product->isActive
						|| !$product->link
						|| $product->link === '#'
						|| $product->priceInBaseCurrency < $minimumPrice
					) {
						continue;
					}

					$urlJoiner = \str_contains($product->link, '?') ? '&' : '?';
					if (Config::load('disableCurrParameter')) {
						if ($this->currency && $this->currency !== $defaultDomainCurrency) {
							$product->link .= $urlJoiner . 'curr=' . $this->currency;
						}
					} else if ($this->currency) {
						$product->link .= $urlJoiner . 'curr=' . $this->currency;
					}

					$category = $product->defaultCategory;

					$catExports = $categoryExports[$product->defaultCategoryId];
					$needLoop   = [];
					$cat        = $category;
					while (count($needLoop) !== 3 && $cat) {
						if (isset($categoryExports[$cat->id])) {
							foreach ($categoryExports[$cat->id] as $service => $data) {
								if (!isset($needLoop[$service])) {
									$catExports[$service] = $data;
								}
								if ($data['status'] !== 2) {
									$needLoop[$service] = false;
								}
							}
						}
						$cat = $cat->getParent();
					}

					if (!$category) {
						continue;
					}

					$product->addExtraField('export', $exports[$product->getId()] ?? []);
					$product->addExtraField('category', $category);
					$product->addExtraField('categoryExports', $catExports);
					$product->addExtraField('disabledByManufacturer', $this->cDisabledManufacturers[$product->manufacturerId] ?? []);

					$result['products'][] = $product;
				}
			}

			$this->em->clear();
			gc_collect_cycles();
		}

		return $result;
	}

	public function getCategories(): array
	{
		if ($this->cCategories === null) {
			$this->cCategories = [];

			$exports = [];
			foreach ($this->em->getRepository(CategoryExport::class)->createQueryBuilder('ce')
				         ->where('ce.lang = :lang')->setParameter('lang', $this->translator->getLocale())
				         ->getQuery()->getArrayResult() as $row) {
				/** @var array $row */
				$exports[$row['id']][$row['service']] = $row;
			}

			$categories = [];
			foreach ($this->em->getRepository(Category::class)->createQueryBuilder('c')
				         ->select('c.id, IDENTITY(c.parent) as parent, c.isPublished, ct.name')
				         ->innerJoin('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->getQuery()->getArrayResult() as $row) {
				/** @var array $row */
				$categories[$row['id']] = $row;
			}

			foreach ($categories as $category) {
				$name      = $category['name'];
				$export    = [];
				$parent    = $categories[$category['parent']];
				$parentIds = [];

				if ($exports[$category['id']]) { // @phpstan-ignore-line
					$export[] = $exports[$category['id']];
				}

				$i = 0;
				while ($parent && $i < 20) {
					$parentIds[] = $parent['id'];
					$name        = $parent['name'] . ' | ' . $name;

					$parent = $categories[$parent['parent']];
					$i++;
				}

				$data = [
					'categoryText' => $name,
					'status'       => $category['isPublished'] ? 1 : 0,
					'exports'      => [],
				];

				foreach ($parentIds as $id) {
					if ($exports[$id]) { // @phpstan-ignore-line
						$export[] = $exports[$id];
					}
				}

				foreach ($export as $y => $vals) {
					foreach ($vals as $k => $v) {
						$data['exports'][$y][$k] = [
							'categoryText' => $v['categoryText'] ?: $data['categoryText'],
							'status'       => $v['status'],
						];
					}
				}

				$this->cCategories[$category['id']] = $data;
			}
		}

		return $this->cCategories;
	}
}
