<?php declare(strict_types = 1);

namespace EshopProductsComparison\FrontModule\Model\Export;

use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\Model\Helpers\ProductConditionEnum;
use EshopProductsComparison\FrontModule\Model\Event\XmlShopItemEvent;
use Nette\Utils\Strings;

class FacebookXmlService extends XmlService
{
	protected string $name = 'facebook';

	public function startExport(): void
	{
		parent::startExport();

		$siteName = $this->domain->siteName;

		$this->writeToFile('<?xml version="1.0" encoding="utf-8"?>' . PHP_EOL .
			'<rss version="2.0" >' . PHP_EOL .
			'<channel>' . PHP_EOL .
			'<title>' . $siteName . '</title>' . PHP_EOL .
			'<link>' . $this->baseUrl . '</link>' . PHP_EOL .
			'<description>Produkty obchodu ' . $siteName . '</description>' . PHP_EOL);
	}

	public function writeItem(Product $product): void
	{
		if (isset($this->usedProducts[$product->getId()])) {
			return;
		}

		$this->usedProducts[$product->getId()] = true;

		$export = $this->prepareExportData($product);

		if (
			$export['status'] === 0
			|| array_key_exists($this->name, (array) ($product->getExtraField('disabledByManufacturer') ?? []))
		) {
			return;
		}

		$currencyEntity = $this->getCurrencyEntity();
		if (!$currencyEntity) {
			return;
		}

		$dom               = new \DOMDocument();
		$dom->formatOutput = true;
		$dom->encoding     = 'utf-8';
		$shopitem          = $dom->createElement('item');

		$child = $dom->createElement('id', (string) $product->getId());
		$shopitem->appendChild($child);

		$child     = $dom->createElement('title');
		$childText = $dom->createTextNode($export['product'] ?: '');
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child       = $dom->createElement('description');
		$description = $export['productDescription'];
		if (!$description) {
			$description = $product->shortDescription;
		}
		$childText = $dom->createCDATASection(Strings::normalize(strip_tags((string) ($description ?: $product->getDescription()))));
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		if ($product->getAvailability()?->canAddToCart()) {
			$fbAv = 'in stock';
		} else {
			$fbAv = 'out of stock';
		}

		$child = $dom->createElement('availability', $fbAv);
		$shopitem->appendChild($child);

		$child = $dom->createElement('condition', ProductConditionEnum::toFacebook($product->condition));
		$shopitem->appendChild($child);

		if ($product->getRetailPrice() > $product->getPrice()) {
			$child = $dom->createElement('sale_price', number_format(round($product->getPrice(), $currencyEntity->decimals), 2, '.', '') . ' ' . $this->currency);
			$shopitem->appendChild($child);
			$child = $dom->createElement('price', number_format(round($product->getRetailPrice(), $currencyEntity->decimals), 2, '.', '') . ' ' . $this->currency);
			$shopitem->appendChild($child);
		} else {
			$child = $dom->createElement('price', number_format(round($product->getPrice(), $currencyEntity->decimals), 2, '.', '') . ' ' . $this->currency);
			$shopitem->appendChild($child);
		}

		$child     = $dom->createElement('link');
		$childText = $dom->createCDATASection((string) $product->link);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		if ($product->getGallery()) {
			$coverId = null;

			if ($product->getGallery()->getCover()) {
				$coverId = (int) $product->getGallery()->getCover()->id;
				$child   = $dom->createElement('image_link', ($this->baseUrl . str_replace(' ', '%20', (string) $product->getGallery()->getCover()->getFilePath())));
				$shopitem->appendChild($child);
			}

			$additionalImages = [];
			$i                = 1;
			foreach ($product->getGallery()->getImages() as $img) {
				if ((int) $img->id === $coverId) {
					continue;
				}

				$additionalImages[] = $this->baseUrl . str_replace(' ', '%20', (string) $img->getFilePath());
				$i++;

				if ($i > 20) {
					break;
				}
			}

			if (!empty($additionalImages)) {
				$child = $dom->createElement('additional_image_link', implode('|', $additionalImages));
				$shopitem->appendChild($child);
			}
		}

		if ($product->getManufacturer() && trim($product->getManufacturer()->name)) {
			$child = $dom->createElement('brand', (string) $product->getManufacturer()->name);
			$shopitem->appendChild($child);
		}

		if ($product->variantId) {
			$param = $dom->createElement('item_group_id', 'G_' . $product->variantId);
			$shopitem->appendChild($param);
		}

		if ($product->getEan()) {
			$child = $dom->createElement('gtin', $product->getEan());
			$shopitem->appendChild($child);
		}

		if ($export['categoryText']) {
			$child = $dom->createElement('fb_product_category', trim((string) $export['categoryText']));
			$shopitem->appendChild($child);
		}

		$child     = $dom->createElement('rich_text_description');
		$childText = $dom->createCDATASection($product->getDescription() ?: '');
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		foreach ($this->getSpeditionsForPrice($product->getPrice(), $product->isOversize, $product->hasDisabledPickupPoints(), $product->disabledSpeditions, $product->disabledPayments) as $country => $speditions) {
			$country = Strings::upper($country);
			if ($country !== Strings::upper($this->domain->getDefaultCountry())) {
				continue;
			}

			$usedSpeds = [];

			foreach ($speditions as $spedName => $spedition) {
				if (isset($usedSpeds[$spedition['speditionName']])) {
					continue;
				}

				$usedSpeds[$spedition['speditionName']] = true;

				$delivery = $dom->createElement(
					'shipping',
					$country . '::' . $spedition['speditionName'] . ':' . number_format(round($spedition['price'] ?: 0, $currencyEntity->decimals), 2, '.', '') . ' ' . $this->currency
				);
				$shopitem->appendChild($delivery);
			}
		}

		if ($product->weight) {
			$child = $dom->createElement('shipping_weight', number_format(($product->weight / 1000), 2, ',', '') . ' kg');
			$shopitem->appendChild($child);
		}

		$usedFeatures = [];
		$colors       = [];
		foreach ($product->getFeatures() as $feature) {
			if (
				!$this->canExportFeature((int) $feature->idFeature)
				|| ($feature->idFeature && array_key_exists($feature->idFeature, $usedFeatures))
				|| !$feature->showInExport
			) {
				continue;
			}

			$featureValue = $this->getExportFeatureValue((int) $feature->idFeatureValue) ?: $feature->value;
			$googleType   = $this->getGoogleFeatureType((int) $feature->idFeature);

			if ($googleType) {
				$value = null;

				if ($googleType === 'color') {
					$colors[] = $featureValue;
					continue;
				}

				if (in_array($googleType, ['material', 'pattern', 'size', 'size_type'])) {
					$value = $featureValue;
				} else if (in_array($googleType, [
					'gender',
					'age_group',
					'is_bundle',
					'energy_efficiency_class',
					'min_energy_efficiency_class',
					'min_energy_efficiency_class',
					'size_system',
				])) {
					$value = $this->getGoogleFeatureValueType((int) $feature->idFeatureValue);
				}

				if ($value) {
					$param = $dom->createElement($googleType, $value);
					$shopitem->appendChild($param);
				}
			}

			$usedFeatures[$feature->idFeature] = $feature->idFeatureValue;
		}

		if (!empty($colors)) {
			$param = $dom->createElement('color', implode('/', $colors));
			$shopitem->appendChild($param);
		}

		foreach ([0, 1, 2, 3, 4] as $cfI) {
			if ($export['customLabel' . $cfI]) {
				$param = $dom->createElement('custom_label_' . $cfI, $export['customLabel' . $cfI]);
				$shopitem->appendChild($param);
			}
		}

		$event = new XmlShopItemEvent($dom, $shopitem, $product, 'facebook');
		$this->eventDispatcher->dispatch($event, 'eshopProductsComparison.xmlShopItem');

		$dom->appendChild($shopitem);

		$this->writeToFile($dom->saveXML($dom->documentElement) . PHP_EOL);
	}

	public function endExport(): void
	{
		$this->writeToFile('</channel>' . PHP_EOL . '</rss>');

		parent::endExport();
	}
}
