<?php declare(strict_types = 1);

namespace EshopSales\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopSales\AdminModule\Model\OrderSales;
use EshopSales\Model\Entities\OrderSale;
use Nette\Utils\ArrayHash;

class OrderSaleForm extends BaseControl
{
	/** @var int @persistent */
	public $id;

	/** @var OrderSale */
	protected $orderSale;

	/** @var OrderSales */
	protected $orderSales;

	public function __construct(OrderSales $orderSales)
	{
		$this->orderSales = $orderSales;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('code', 'eshopSales.orderSale.code')->setNullable();
		$form->addText('amount', 'eshopSales.orderSale.amount')->setType('number')
			->setAttribute('min', 0)->setAttribute('step', 1)
			->setRequired();
		$form->addSelect('type', 'eshopSales.orderSale.type', OrderSale::getTypesOptions())->setRequired();
		$form->addText('fromPrice', 'eshopSales.orderSale.fromPrice')->setType('number')
			->setAttribute('min', 0)->setAttribute('step', 1)
			->setRequired();
		$form->addDateTimePicker('dateFrom', 'eshopSales.orderSale.dateFrom')->setNullable();
		$form->addDateTimePicker('dateTo', 'eshopSales.orderSale.dateTo')->setNullable();
		$form->addBool('isActive', 'default.isActive')->setDefaultValue(1);

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$sale = $this->id
				? $this->orderSales->get($this->id)
				: new OrderSale($values->type, (float) $values->amount, (float) $values->fromPrice);

			$sale->setType($values->type);
			$sale->setDateFrom($values->dateFrom);
			$sale->setDateTo($values->dateTo);
			$sale->setAmount((float) $values->amount);
			$sale->setFromPrice((float) $values->fromPrice);
			$sale->code     = $values->code;
			$sale->isActive = $values->isActive;

			$this->em->persist($sale)->flush();
			$this->getPresenter()->flashMessageSuccess('default.saved');
			$form->addCustomData('orderSaleId', $sale->getId());
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setOrderSale(int $id): void
	{
		$this->orderSale = $this->orderSales->get($id);

		if (!$this->orderSale)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);

		$s = $this->orderSale;
		$d = [
			'code'      => $s->code,
			'amount'    => $s->getAmount(),
			'fromPrice' => $s->getFromPrice(),
			'dateFrom'  => $s->getDateFrom(),
			'dateTo'    => $s->getDateTo(),
			'isActive'  => (int) $s->isActive,
		];

		if ($s->getType() && array_key_exists($s->getType(), $this['form']['type']->getItems()))
			$d['type'] = $s->getType();

		$this['form']->setDefaults($d);
	}
}
