<?php declare(strict_types = 1);

namespace EshopSales\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopSales\AdminModule\Model\OrderSales;
use EshopSales\Model\Entities\OrderSale;

class OrderSalesGrid extends BaseControl
{
	/** @var OrderSales */
	protected $orderSales;

	public function __construct(OrderSales $orderSales)
	{
		$this->orderSales = $orderSales;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();

		if ($this->orderSales->remove($id))
			$presenter->flashMessageSuccess('default.deleted');
		else
			$presenter->flashMessageDanger('default.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->orderSales->getEr()->createQueryBuilder('os')
			->orderBy('os.fromPrice', 'ASC');

		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnText('code', 'eshopSales.orderSale.code');
		$grid->addColumnText('amount', 'eshopSales.orderSale.amount')->setRenderer(function(OrderSale $row) {
			$type = OrderSale::TYPES[$row->getType()];
			if ($row->getType() == OrderSale::TYPE_FIX)
				return number_format($row->amount, 2, ',', ' ');
			else
				return (int) $row->amount . $type['symbol'];
		})->setAlign('right');
		$grid->addColumnNumber('fromPrice', 'eshopSales.orderSale.fromPrice')->setFormat(2, ',', ' ');
		$grid->addColumnDateTime('dateFrom', 'eshopSales.orderSale.dateFrom')->setFormat('j. n. Y H:i:s');
		$grid->addColumnDateTime('dateTo', 'eshopSales.orderSale.dateTo')->setFormat('j. n. Y H:i:s');
		$grid->addColumnStatus('isActive', 'default.isActive')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Prototype
		$grid->getColumn('isActive')->getElementPrototype('td')->class[] = 'w1nw';

		// Actions
		$grid->addAction('edit', '', 'editOrderSale!')->setIcon('fas fa-pencil-alt')->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setIcon('fas fa-times')->setBsType('danger')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->orderSales->setActive($id, $newStatus))
			$presenter->flashMessageSuccess('default.publishChanged');
		else
			$presenter->flashMessageDanger('default.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}
}
