<?php declare(strict_types = 1);

namespace EshopSales\AdminModule\Components\Order;

use Core\AdminModule\Model\Sites;
use Core\Model\UI\AdminPresenter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\SelectInput;
use Currency\AdminModule\Model\Currencies;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeatureValues;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\AdminModule\Model\GroupsCustomers;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use EshopSales\AdminModule\Model\OrderSales;
use EshopSales\Model\Config;
use EshopSales\Model\Entities\OrderSale;
use EshopSales\Model\EshopSalesConfig;
use Exception;
use Nette\Application\Attributes\Persistent;
use Nette\Application\LinkGenerator;
use Nette\Application\UI\Presenter;
use Nette\Caching\Cache;
use Nette\Utils\ArrayHash;
use Nette\Utils\Arrays;
use Nette\Utils\DateTime;

class OrderSaleForm extends BaseControl
{
	#[Persistent]
	public ?int $id = null;

	protected ?OrderSale $orderSale = null;

	public function __construct(
		protected OrderSales        $orderSales,
		protected Config            $config,
		protected Sites             $adminSites,
		protected \Core\Model\Sites $sites,
		protected Features          $featuresService,
		protected FeatureValues     $featureValuesService,
		protected Manufacturers     $manufacturers,
		protected Categories        $categories,
		protected GroupsCustomers   $customersGroups,
		protected EshopOrdersCache  $eshopOrdersCache,
		protected Currencies        $currencies,
		protected Products          $products,
		protected LinkGenerator     $linkGenerator,
	)
	{
	}

	public function render(): void
	{
		$currencies = [];
		foreach ($this->currencies->getAll() as $currency) {
			if (!$currency->isActive || $currency->getCode() === \Currency\Model\Config::load('default')) {
				continue;
			}

			$key                                    = $currency->site ? $currency->site->getIdent() : '';
			$currencies[$key][$currency->getCode()] = [
				'rate' => (float) $currency->rate,
			];
		}

		$this->template->edit            = $this->orderSale instanceof OrderSale;
		$this->template->fastSalesConfig = EshopSalesConfig::load('fastSales');
		$this->template->currencies      = $currencies;
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax()
			->setShowLangSwitcher(false);

		$form->addText('code', 'eshopSales.orderSale.code')->setNullable();
		$form->addText('amount', 'eshopSales.orderSale.amount')
			->setHtmlType('number')
			->setHtmlAttribute('min', 0)
			->setHtmlAttribute('step', 1)
			->setRequired();
		$type = $form->addSelect('type', 'eshopSales.orderSale.type', OrderSale::getTypesOptions())->setRequired();
		$form->addText('fromPrice', 'eshopSales.orderSale.fromPrice')
			->setHtmlType('number')
			->setHtmlAttribute('min', 0)
			->setHtmlAttribute('step', 1)
			->addConditionOn($type, $form::EQUAL, OrderSale::TYPE_FIX)
			->addRule($form::MIN, null, $form['amount']);
		$form->addDatePicker('dateFrom', 'eshopSales.orderSale.dateFrom');
		$form->addDatePicker('dateTo', 'eshopSales.orderSale.dateTo');
		$form->addTextArea('description', 'eshopSales.orderSale.description');

		$form->addInteger('repetitions', 'eshopSales.orderSale.repetitions')->setNullable();

		$form->addBool('isActive', 'default.isActive')->setDefaultValue(1);

		$sites = $this->adminSites->getOptionsForSelect();
		if (count($sites) > 1) {
			$form->addSelect('site', 'eshopSales.orderSale.site', [null => null] + $sites);
		} else {
			$form->addHidden('site', array_values($sites)[0]);
		}

		$bulkCreation = $form->addBool('bulkCreation', 'eshopSales.orderSale.bulkCreation.caption');
		$form->addText('bulkCreationCount', 'eshopSales.orderSale.bulkCreationCount')
			->setNullable()
			->setHtmlType('number')
			->addConditionOn($bulkCreation, $form::EQUAL, '1')
			->setRequired()
			->addRule($form::INTEGER);

		if (EshopSalesConfig::load('allowConditions')) {
			$form->addCheckboxList('customerGroups', 'eshopSales.orderSale.customerGroups', $this->customersGroups->getOptionsForSelect());

			$sites = $this->adminSites->getAll();
			if (count($sites) > 1) {
				$flat = [];
				foreach ($sites as $site) {
					$tmp = $this->categories->getFlatTree($site->getIdent());

					if ($tmp === []) {
						continue;
					}

					$flat[] = [[
						'id'     => $tmp[0]['parent'],
						'parent' => 0,
						'name'   => $site->getIdent(),
					]];

					$flat[] = $tmp;
				}
				$flat = array_merge(...$flat);
			} else {
				$flat = $this->categories->getFlatTree();
			}
			$form->addCheckboxNestedList('categories', 'eshopSales.orderSale.categories', $flat);
			$form->addCheckboxList('manufacturers', 'eshopSales.orderSale.manufacturers', $this->manufacturers->getOptionsForSelect());
			$form->addCheckboxNestedList('features', 'eshopSales.orderSale.features', $this->featuresService->getFlatTree(false));

			$productLink = $this->linkGenerator->link('EshopCatalog:Admin:Products:edit', ['__ID__']);

			$form->addText('product', 'eshopSales.orderSale.selectedProducts')
				->setHtmlAttribute('data-autocomplete-name', 'saleProduct')
				->setHtmlAttribute('data-autocomplete-keys', 'id,code1,ean,name')
				->setHtmlAttribute('data-product-link', $productLink)
				->setOmitted();

			$form->monitor(AdminPresenter::class, function(AdminPresenter $presenter) use ($form): void {
				$form->getComponent('product')
					->setHtmlAttribute('data-autocomplete-url', $this->linkGenerator->link('EshopCatalog:Cron:Products:loadAll', [
						'excluded' => [$presenter->getParameter('id')],
					]));
			});

			$form->addCustomData('productLink', $productLink);
		}

		$form->addHidden('orderSaleId');

		$form->addSaveCancelControl();
		$validationScope = iterator_to_array($form->getControls());

		if (EshopSalesConfig::load('allowConditions')) {
			unset(
				$validationScope['customerGroups'],
				$validationScope['categories'],
				$validationScope['manufacturers'],
				$validationScope['features']
			);
		}
		$form->getComponent('saveControl')['save']->setValidationScope($validationScope);
		$form->getComponent('saveControl')['saveAndClose']->setValidationScope($validationScope);

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$presenter = $this->presenter;

		if ($this->id) {
			$this->orderSale = $this->orderSales->get($this->id);
		}

		try {
			$this->em->beginTransaction();

			$bulkCreationCount = $this->id ? 1 : $values->bulkCreationCount ?? 1;

			$httpData = (array) $form->getHttpData();

			$products = $httpData['product'] ?? [];

			$saved = $this->orderSales->save([
				'type'                          => $values->type,
				'dateFrom'                      => $values->dateFrom ? DateTime::from($values->dateFrom->getTimestamp()) : null,
				'dateTo'                        => $values->dateTo ? DateTime::from($values->dateTo->getTimestamp()) : null,
				'amount'                        => (float) $values->amount,
				'fromPrice'                     => (float) $values->fromPrice,
				'isActive'                      => $values->isActive,
				'code'                          => $values->code,
				'maxRepetitions'                => $values->repetitions,
				'bulkCreationCount'             => $bulkCreationCount,
				'orderItem'                     => $this->orderSale && $this->orderSale->orderItem ? $this->orderSale->orderItem : null,
				'editedId'                      => $this->id,
				'autoGenerateCodeIfCodeIsEmpty' => false,
				'description'                   => $values->description,
				'sites'                         => $values->site ? [$values->site] : [],
				'customerGroups'                => $values->customerGroups ?: null,
				'categories'                    => $values->categories ?: null,
				'manufacturers'                 => $values->manufacturers ?: null,
				'features'                      => $values->features ?: null,
				'products'                      => $products ? array_values($products) : null,
			]);

			if ($bulkCreationCount === 1 && ($sale = Arrays::first($saved))) {
				$form->addCustomData('orderSaleId', $sale->getId());
			}

			$this->em->commit();
			$presenter->flashMessageSuccess('default.saved');

			$this->eshopOrdersCache->getCache()->clean([
				Cache::Tags => ['orderSales'],
			]);

			$this->em->beginTransaction();
		} catch (Exception $e) {
			$this->em->rollback();
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setOrderSale(int $id): void
	{
		$presenter = $this->getPresenter(false);
		if (!$presenter instanceof Presenter) {
			return;
		}

		$this->orderSale = $this->orderSales->get($id);

		if (!$this->orderSale) {
			$presenter->error();
		}
		$s = $this->orderSale;

		$d = [
			'code'           => $s->code,
			'amount'         => $s->getAmount(),
			'fromPrice'      => $s->getFromPrice(),
			'dateFrom'       => $s->getDateFrom(),
			'dateTo'         => $s->getDateTo(),
			'repetitions'    => $s->maxRepetitions,
			'description'    => $s->description,
			'isActive'       => (int) $s->isActive,
			'orderSaleId'    => $id,
			'site'           => $s->sites->toArray() ? Arrays::first($s->sites->toArray())->site->getIdent() : null,
			'customerGroups' => $s->getCustomerGroups(),
			'categories'     => $s->getCategories(),
			'manufacturers'  => $s->getManufacturers(),
			'features'       => $s->getFeatures(),
		];

		/** @var SelectInput $typeControl */
		$typeControl = $this['form']['type'];
		if ($s->getType() && array_key_exists($s->getType(), $typeControl->getItems())) {
			$d['type'] = $s->getType();
		}

		$this['form']->setDefaults($d);

		$productsDefault = [];
		if ($s->getProducts()) {
			foreach ($this->em->getRepository(Product::class)->createQueryBuilder('p')
				         ->select('p.id, p.code1, t.name')
				         ->innerJoin('p.productTexts', 't', Join::WITH, 't.lang = :lang')
				         ->andWhere('p.id IN (:ids)')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->setParameter('ids', $s->getProducts())
				         ->getQuery()->getScalarResult() as $row) {
				/** @var array $row */
				$productsDefault[$row['id']] = [
					'code1' => $row['code1'],
					'name'  => $row['name'],
				];
			}
		}
		$this['form']->addCustomData('productsDefault', $productsDefault);

		$this->redrawControl('form');
	}

	public function handleFillFastSale(string $fastSaleId): void
	{
		$eshopSalesConfig = (array) EshopSalesConfig::load('fastSales');

		if (!$eshopSalesConfig['enable']) {
			return;
		}

		$fastSales = $eshopSalesConfig['items'];
		$fastSales = \Core\Model\Helpers\Arrays::search($fastSales, static fn($k, $v): bool => $v['id'] === $fastSaleId, false);

		if (!isset($fastSales[0])) {
			return;
		}

		$fastSale    = $fastSales[0];
		$value       = (float) $fastSale['value'];
		$now         = new DateTime;
		$expiration  = $eshopSalesConfig['expiration'];
		$repetitions = $eshopSalesConfig['repetitions'];

		$d = [
			'code'        => $this->orderSales->generateCode(),
			'amount'      => (float) $fastSale['value'],
			'fromPrice'   => $value + 1,
			'dateFrom'    => $now,
			'dateTo'      => $now->modifyClone("+ {$expiration} days"),
			'repetitions' => $repetitions,
			'isActive'    => 1,
			'type'        => $fastSale['type'],
		];
		$this['form']->setDefaults($d);
		$this->redrawControl('form');
	}

}
