<?php declare(strict_types = 1);

namespace EshopSales\AdminModule\Components\Order;

use Core\AdminModule\Model\Sites;
use Core\Model\Entities\QueryBuilder;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Doctrine\DBAL\ArrayParameterType;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Utils\Helpers;
use EshopSales\AdminModule\Model\UsedOrderSales;
use EshopSales\Model\Entities\OrderSale;
use EshopSales\Model\Entities\UsedOrderSale;
use EshopSales\Model\EshopSalesConfig;
use Nette\Utils\Html;

class UsedOrderSalesGrid extends BaseControl
{
	public function __construct(
		protected UsedOrderSales               $usedOrderSales,
		protected Sites                        $adminSites,
		protected Helpers                      $helpers,
		protected Orders                       $orders,
		protected IOrderSalesStatisticsFactory $orderSalesStatisticsFactory,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentOrderSalesStatistics(): OrderSalesStatistics
	{
		return $this->orderSalesStatisticsFactory->create();
	}

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		if (EshopSalesConfig::load('usedOrderSalesGrid.csvExport')) {
			$grid->addExportCsv('CSV Export', 'slevy.csv', 'windows-1250');
		}

		$qb = $this->usedOrderSales->getEr()->createQueryBuilder('os');
		$qb->addSelect('osos.created')
			->join('os.order', 'oso')
			->join('oso.orderStatuses', 'osos')
			->join('osos.status', 'oss', 'WITH', "oss.id = 'created'");

		$dataSource = new DoctrineDataSource($qb, 'os.id');

		/** @var Order[] $orders */
		$orders  = [];
		$inSites = [];

		$dataSource->onDataLoaded[] = function(array $data) use (&$orders, &$inSites): void {
			/** @var array[] $data */

			$conn      = $this->em->getConnection();
			$ids       = [];
			$ordersIds = [];

			foreach ($data as $r) {
				$row   = $r[0];
				$ids[] = $row->getId();

				if ($row->order) {
					$ordersIds[] = $row->order->getId();
				}
			}

			if ($ordersIds !== []) {
				foreach ($this->orders->getFullOrderQb()
					         ->addSelect('ois')
					         ->leftJoin('items.sales', 'ois')
					         ->andWhere('o.id IN (:id)')
					         ->setParameter('id', $ordersIds)
					         ->getQuery()->getResult() as $row) {
					$orders[$row->getId()] = $row;
				}
			}

			if ($ids !== []) {
				foreach ($conn->executeQuery("SELECT DISTINCT used_order_sale_id, site_id FROM eshop_sales__used_order_sale_in_site WHERE used_order_sale_id IN (:ids)", ['ids' => $ids], ['ids' => ArrayParameterType::INTEGER])
					         ->iterateAssociative() as $v) {
					$inSites[$v['used_order_sale_id']][] = $v['site_id'];
				}
			}
		};

		$grid->setDataSource($dataSource);
		$grid->setDefaultPerPage(20);
		$grid->defaultSort = ['dateFrom' => 'asc'];

		// Columns
		$grid->addColumnText('code', 'eshopSales.orderSale.code')
			->setRenderer(function(array $data) {
				/** @var UsedOrderSale $entity */
				$entity = $data[0];

				return $entity->code ?? '';
			})
			->setFilterText();
		$grid->addColumnText('type', 'eshopSales.orderSale.type')
			->setRenderer(function(array $data) {
				/** @var UsedOrderSale $entity */
				$entity = $data[0];

				return $this->t(OrderSale::getTypesOptions()[$entity->type]);
			})
			->setFilterSelect(['' => ''] + OrderSale::getTypesOptions())
			->setTranslateOptions();
		$grid->addColumnText('amount', 'eshopSales.orderSale.amount')
			->setRenderer(function(array $data) {
				/** @var UsedOrderSale $entity */
				$entity = $data[0];

				return $this->helpers->formatDiscount($entity);
			})
			->setFilterText();
		$grid->getColumn('amount')->setAlign('right');
		$grid->addColumnPrice('orderPrice', 'eshopSales.orderSale.orderPrice')
			->setRenderer(function(array $data) use (&$orders) {
				/** @var UsedOrderSale $entity */
				$entity = $data[0];
				if ($entity->order && isset($orders[$entity->order->getId()])) {
					return $orders[$entity->order->getId()]->getPrice();
				}

				return '';
			});
		$grid->addColumnDateTime('dateFrom', 'eshopSales.orderSale.dateFrom')
			->setRenderer(function(array $data) {
				/** @var UsedOrderSale $entity */
				$entity = $data[0];

				return $entity->dateFrom ? $entity->dateFrom->format('j. n. Y') : '';
			})
			->setSortable();
		$grid->addColumnDateTime('dateTo', 'eshopSales.orderSale.dateTo')
			->setRenderer(function(array $data) {
				/** @var UsedOrderSale $entity */
				$entity = $data[0];

				return $entity->dateTo ? $entity->dateTo->format('j. n. Y') : '';
			})
			->setSortable();
		$grid->addColumnDateTime('created', 'eshopSales.orderSale.applied', 'osos.created')
			->setFitContent()
			->setRenderer(function(array $data) {
				/** @var \DateTimeInterface|null $data */
				$data = $data['created'];
				if ($data) {
					return $data->format('d.m.Y');
				}

				return '';
			})
			->setSortable()
			->setFilterDateRange();
		$grid->addColumnText('order', 'eshopSales.orderSale.appliedIn')
			->setFitContent()
			->setAlign('center')
			->setTemplateEscaping(false)
			->setRenderer(function(array $data) {
				/** @var UsedOrderSale $entity */
				$entity = $data[0];
				$html   = Html::el();
				if ($entity->order !== null) {
					$oId = $entity->order->getId();
					$html->addHtml(
						Html::el('a', [
							'href'   => $this->presenter->link(':EshopOrders:Admin:Default:editOrder', $oId),
							'target' => '_blank',
						])->setText($oId)
					);
				}

				return $html;
			})->setFilterText()
			->setCondition(function(QueryBuilder $qb, $value): void {
				$qb->andWhere('os.order = :order')->setParameter('order', $value);
			});

		if (count($this->adminSites->getAll()) > 1) {
			$grid->addColumnText('sites', 'eshopSales.orderSale.sites')
				->setRenderer(function(array $data) use (&$inSites) {
					/** @var UsedOrderSale $entity */
					$entity = $data[0];

					return implode(', ', $inSites[$entity->getId()] ?? []);
				})
				->setFilterSelect(['' => ''] + $this->adminSites->getOptionsForSelect())
				->setCondition(function(QueryBuilder $qb, $value) {
					$qb->innerJoin('os.sites', 'osSite', Join::WITH, 'osSite.site = :site')
						->setParameter('site', $value);
				});
		}

		if (EshopSalesConfig::load('orderSalesGrid.showDescription')) {
			$grid->addColumnText('description', 'eshopSales.orderSale.description')
				->setRenderer(function(array $data) {
					/** @var UsedOrderSale $entity */
					$entity = $data[0];

					return $entity->description ?? '';
				})
				->setSortable()
				->setFilterText();
		}

		$grid->addAction('delete', '', 'remove!')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	public function handleRemove(int $id): void
	{
		if ($this->usedOrderSales->remove($id)) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.error');
		}

		$this->presenter->redrawControl('flashes');
		$this['grid']->reload();
	}
}
