<?php declare(strict_types = 1);

namespace EshopSales\AdminModule\Model;

use Contributte\Translation\Translator;
use Core\Model\Sites;
use Ecomail;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use EshopSales\Model\EshopSalesConfig;
use Nette\Caching\Cache;
use Nette\DI\Container;
use Nette\Utils\DateTime;
use Tracy\Debugger;

class HolidaysExport
{
	public function __construct(
		protected Container        $container,
		protected OrderSales       $orderSales,
		protected EshopOrdersCache $eshopOrdersCache,
	)
	{
	}

	public function exportToEcoMail(string $siteIdent, string $lang): void
	{
		Sites::$currentIdentOverride = $siteIdent;
		Sites::$currentLangOverride  = $lang;

		/** @var ?Translator $translator */
		$translator = $this->container->hasService('translation.translator') ? $this->container->getService('translation.translator') : null;
		if ($translator) {
			$translator->setLocale($lang);
		}

		$ecoMailFolderId = EshopSalesConfig::load('holidaysCoupon.ecoMailFolderId');
		$enabled         = EshopSalesConfig::load('holidaysCoupon.enableEcomail');

		if (
			!$ecoMailFolderId
			|| !$enabled
			|| !class_exists('MeasuringCodes\FrontModule\Model\TypesList')
			|| !$this->container->hasService('measuringCodes.front.typesList')
		) {
			return;
		}


		/** @var \MeasuringCodes\FrontModule\Model\TypesList $typesList */
		$typesList = $this->container->getService('measuringCodes.front.typesList');

		$ecoMailType = $typesList->getType('ecoMail');
		if (!$ecoMailType || !$ecoMailType->isActive() || !$ecoMailType->getFieldValue('appId') || !$ecoMailType->getFieldValue('listId')) {
			return;
		}

		$listId        = $ecoMailType->getFieldValue('listId');
		$salesFolderId = $ecoMailType->getFieldValue('salesFolderId') ?: 1;
		$ecoMailApi    = new Ecomail($ecoMailType->getFieldValue('appId'));

		$today         = (new \DateTime())->format('m-d');
		$todayNameDays = [];

		$currentPage = 1;

		do {
			$subscribers = (array) $ecoMailApi->page($currentPage)->withQuery('per_page', 1000)->getSubscribers($listId);
			$lastPage    = $subscribers['last_page'] ?? 1;

			if (isset($subscribers['error'])) {
				foreach ($subscribers['message']->errors as $err) {
					Debugger::log('ERROR - ' . $err, 'ecomailHolidayExport');
				}

				return;
			}

			foreach ($subscribers['data'] ?? [] as $subscriber) {
				$nameDay = $subscriber['subscriber']['nameday'] ?? null;

				if ($nameDay && $nameDay === $today) {
					$todayNameDays[] = $subscriber['subscriber']['name'];
				}
			}
		} while ($currentPage++ < $lastPage);

		if ($todayNameDays === []) {
			return;
		}

		$codesToExport = [];

		$saleType   = EshopSalesConfig::loadScalar('holidaysCoupon.type');
		$expiration = EshopSalesConfig::loadScalar('holidaysCoupon.expiration');
		$amount     = (float) EshopSalesConfig::loadScalar('holidaysCoupon.amount');
		$fromPrice  = (float) EshopSalesConfig::loadScalar('holidaysCoupon.fromPrice');

		$currYear = date('Y');
		$dateFrom = DateTime::createFromFormat('Y-m-d', $currYear . '-' . $today);
		if (!$dateFrom) {
			return;
		}

		$dateFrom->setTime(0, 0, 0);
		$dateTo = (($dateFrom)->modifyClone("+{$expiration} days"))->setTime(23, 59, 59);

		foreach ($todayNameDays as $email) {
			foreach ($this->orderSales->save([
				'type'                          => $saleType,
				'dateFrom'                      => $dateFrom,
				'dateTo'                        => $dateTo,
				'amount'                        => $amount,
				'fromPrice'                     => $fromPrice,
				'isActive'                      => true,
				'code'                          => null,
				'maxRepetitions'                => 1,
				'bulkCreationCount'             => 1,
				'orderItem'                     => null,
				'editedId'                      => null,
				'autoGenerateCodeIfCodeIsEmpty' => true,
				'description'                   => $email,
			]) as $row) {
				$codesToExport[] = $row->code;
			}
		}

		$this->eshopOrdersCache->getCache()->clean([
			Cache::Tags => ['orderSales'],
		]);

		if ($codesToExport !== []) {
			Debugger::log(json_encode([
				'codesToExport'     => count($codesToExport),
				'todayNameDays'     => count($todayNameDays),
				'subscribersListId' => $listId,
				'salesFolderId'     => $salesFolderId,
				'validUntil'        => $dateTo->format('Y-m-d'),
			]), '_ecomailHolidayExportLog');

			$ecoMailApi->importCoupons([
				'folderId'        => $salesFolderId,
				'codes'           => $codesToExport,
				'valid_until'     => $dateTo->format('Y-m-d'),
				'update_existing' => false,
			]);
		}
	}
}
