<?php declare(strict_types = 1);

namespace EshopSales\AdminModule\Model\Mappers\OrderSale;

use Contributte\Translation\Translator;
use EshopSales\Model\Entities\OrderSale;
use EshopSales\Model\Entities\OrderSaleInSite;
use Override;

class CsvMapper implements ICsvMapper
{
	public function __construct(
		protected Translator $translator,
	)
	{
	}

	#[Override]
	public function normalize(array $fileContent): array
	{
		// first row is header
		$fileContent = array_slice($fileContent, 1);
		$data        = [];

		foreach ($fileContent as $row) {
			$r = [];
			for ($i = 0, $fieldCount = count($row); $i < $fieldCount; $i++) {
				$r[self::getStructure()[$i]] = $row[$i];
			}
			$r['isActive']                      = 1;
			$r['amount']                        = (float) $r['amount'];
			$r['fromPrice']                     = (float) $r['fromPrice'];
			$r['bulkCreationCount']             = 1;
			$r['autoGenerateCodeIfCodeIsEmpty'] = false;
			$r['editedId']                      = null;
			$r['orderItem']                     = null;
			$r['maxRepetitions']                = empty($r['maxRepetitions']) ? 1 : (int) $r['maxRepetitions'];
			$data[]                             = $r;
		}

		return $data;
	}

	/**
	 * @param OrderSale[] $orderSales
	 */
	#[Override]
	public function toRaw(array $orderSales): string
	{
		$data = [];

		// first row is header
		foreach (self::getStructure() as $col) {
			$data[0][$col] = $this->translator->translate('eshopSales.exportCols.' . $col);
		}

		foreach ($orderSales as $orderSale) {
			$r = [];
			/** @var OrderSaleInSite|null $site */
			$site = array_values($orderSale->sites->toArray())[0];

			foreach (self::getStructure() as $col) {
				$r[$col] = null;
			}

			$r['code']           = $orderSale->code;
			$r['sites']          = $site ? $site->site->getIdent() : null;
			$r['fromPrice']      = $orderSale->getFromPrice();
			$r['maxRepetitions'] = $orderSale->maxRepetitions;
			$r['description']    = $orderSale->description;
			$r['dateFrom']       = $orderSale->getDateFrom() ? $orderSale->getDateFrom()->format('d.m.Y') : null;
			$r['dateTo']         = $orderSale->getDateTo() ? $orderSale->getDateTo()->format('d.m.Y') : null;
			$r['type']           = $orderSale->getType();
			$r['amount']         = $orderSale->getAmount();
			$data[]              = $r;
		}

		$csv = '';
		foreach ($data as $v) {
			$length = count($v);
			$i      = 0;
			foreach ($v as $col) {
				$csv .= '"' . $col . '"';

				if ($i === ($length - 1)) {
					$csv .= PHP_EOL;
				} else {
					$csv .= ';';
				}
				$i++;
			}
		}

		return $csv;
	}

	protected static function getStructure(): array
	{
		return ['code', 'sites', 'fromPrice', 'maxRepetitions', 'description', 'dateFrom', 'dateTo', 'type', 'amount'];
	}

}
