<?php declare(strict_types = 1);

namespace EshopSales\FrontModule\Model\Dao;

use Core\Model\Helpers\Arrays;
use DateTimeInterface;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopOrders\Model\Entities\Customer;
use EshopSales\Model\EshopSalesConfig;
use Nette\Utils\Validators;

class OrderSale
{
	/** @var float */
	public $value;

	/** @var float */
	public $valueInBaseCurrency;

	/** @var float */
	public $fromPrice;

	/** @var float */
	public $fromPriceInBaseCurrency;

	public array              $customerGroups = [];
	public array              $categories     = [];
	public array              $manufacturers  = [];
	public array              $features       = [];
	public array              $products       = [];
	public ?string            $description    = null;
	public ?DateTimeInterface $dateFrom       = null;
	public ?DateTimeInterface $dateTo         = null;

	public function __construct(
		public int     $id,
		public ?string $code,
		public string  $type,
	)
	{
	}

	public function isAutoSale(): bool { return Validators::isNone($this->code); }

	public function getCode(): ?string
	{
		return $this->code ?: $this->getAutoSaleCode();
	}

	public function getAutoSaleCode(): ?string
	{
		$prefix = $this->isAutoSale() ? EshopSalesConfig::loadScalar('autoSales.prefix') : null;
		if (!$prefix) {
			return null;
		}

		if (EshopSalesConfig::load('autoSales.addValue')) {
			$prefix .= $this->value;
		}

		return (string) $prefix;
	}

	public function typeData(): ?array { return \EshopSales\Model\Entities\OrderSale::TYPES[$this->type] ?? null; }

	public function canUseOnProduct(Product $product, ?Customer $customer = null): bool
	{
		if (EshopSalesConfig::load('allowConditions')) {
			if ($this->customerGroups && (!$customer || !$customer->getGroupCustomers() || !in_array($customer->getGroupCustomers()->getId(), $this->customerGroups))) {
				return false;
			}

			if ($this->categories && (!in_array($product->defaultCategoryId, $this->categories) && !array_intersect($this->categories, $product->categories))) {
				return false;
			}

			if ($this->manufacturers && !in_array($product->manufacturerId, $this->manufacturers)) {
				return false;
			}

			if ($this->features) {
				$tmp = array_intersect($this->features, $product->featureValuesIds);
				if (count($tmp) !== count($this->features)) {
					return false;
				}
			}

			if ($this->products && !Arrays::contains($this->products, $product->getId())) {
				return false;
			}
		}

		return true;
	}

	public function getSortPosition(): int
	{
		return match ($this->type) {
			\EshopSales\Model\Entities\OrderSale::TYPE_PERCENT => 1,
			\EshopSales\Model\Entities\OrderSale::TYPE_FIX => 2,
			default => 9999,
		};
	}
}
