<?php declare(strict_types = 1);

namespace EshopSales\AdminModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TActive;
use Core\Model\Mailing\TemplateFactory;
use Core\Model\Sites;
use DateTimeInterface;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderGift;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use EshopOrders\Model\Utils\Helpers;
use EshopSales\Model\Entities\OrderSale;
use EshopSales\Model\Entities\OrderSaleInSite;
use EshopSales\Model\Entities\UsedOrderSale;
use EshopSales\Model\EshopSalesConfig;
use EshopSales\Model\Pdf\IDiscountCouponPdfBuilderFactory;
use Exception;
use Mpdf\Output\Destination;
use Nette\Caching\Cache;
use Contributte\Translation\Translator;
use Nette\Mail\Mailer;
use Nette\Mail\Message;
use Nette\Schema\Expect;
use Nette\Schema\Processor;
use Nette\Utils\ArrayHash;
use Nette\Utils\Arrays;
use Nette\Utils\DateTime;
use Nette\Utils\Json;
use Nette\Utils\Random;
use Tracy\Debugger;

/**
 * @method OrderSale|null get($id)
 */
class OrderSales extends BaseEntityService
{
	use TActive;

	/** @var string */
	protected $entityClass = OrderSale::class;

	protected Sellers                          $sellers;
	protected Sites                            $sites;
	protected \Core\AdminModule\Model\Sites    $adminSites;
	protected Mailer                           $mailer;
	protected TemplateFactory                  $mailTemplateFactory;
	protected Translator                       $translator;
	protected IDiscountCouponPdfBuilderFactory $discountCouponPdfBuilderFactory;
	protected EshopOrdersCache                 $eshopOrdersCache;

	public function __construct(
		Sellers                          $sellers,
		Sites                            $sites,
		Mailer                           $mailer,
		TemplateFactory                  $mailTemplateFactory,
		Translator                       $translator,
		IDiscountCouponPdfBuilderFactory $discountCouponPdfBuilderFactory,
		\Core\AdminModule\Model\Sites    $adminSites,
		EshopOrdersCache                 $eshopOrdersCache
	)
	{
		$this->sellers                         = $sellers;
		$this->sites                           = $sites;
		$this->mailer                          = $mailer;
		$this->mailTemplateFactory             = $mailTemplateFactory;
		$this->translator                      = $translator;
		$this->discountCouponPdfBuilderFactory = $discountCouponPdfBuilderFactory;
		$this->adminSites                      = $adminSites;
		$this->eshopOrdersCache                = $eshopOrdersCache;
	}

	/** @return array<int, OrderSale> */
	public function save(array $data): array
	{
		$data['sites']       = !is_array($data['sites']) && empty($data['sites']) ? [] : $data['sites'];
		$data['sites']       = !is_array($data['sites']) && !empty($data['sites']) ? [$data['sites']] : $data['sites'];
		$allowedSiteValues   = array_values($this->adminSites->getOptionsForSelect());
		$allowedSiteValues[] = null;
		$allowedSiteValues[] = '';
		$dateHandler         = function($v) {
			if (empty($v)) {
				return null;
			}

			if ($v instanceof DateTimeInterface || !is_string($v)) {
				return $v;
			}

			try {
				return new DateTime($v);
			} catch (Exception $exception) {
				return $v;
			}
		};
		(new Processor)->process(Expect::structure([
			'type'                          => Expect::anyOf(...array_keys(OrderSale::TYPES))->required(),
			'dateFrom'                      => Expect::type(DateTime::class)->before($dateHandler)->nullable(),
			'dateTo'                        => Expect::type(DateTime::class)->before($dateHandler)->nullable(),
			'amount'                        => Expect::float()->required()->min(0),
			'fromPrice'                     => $data['type'] === OrderSale::TYPE_FIX
				? Expect::float()->required()->min((float) $data['amount'])
				: Expect::float()->nullable(),
			'isActive'                      => Expect::anyOf(true, false, 1, 0)->required(),
			'code'                          => Expect::string()->nullable(),
			'maxRepetitions'                => Expect::int()->nullable(),
			'bulkCreationCount'             => Expect::int()->required()->min(1),
			'orderItem'                     => Expect::type(OrderItem::class)->nullable(),
			'orderGift'                     => Expect::type(OrderGift::class)->nullable(),
			'editedId'                      => Expect::type('string|int|null')->nullable(),
			'autoGenerateCodeIfCodeIsEmpty' => Expect::anyOf(true, false, 1, 0)->required(),
			'description'                   => Expect::string()->nullable(),
			'sites'                         => Expect::listOf(Expect::anyOf(...$allowedSiteValues))->default([]),
			'customerGroups'                => Expect::array()->nullable(),
			'categories'                    => Expect::array()->nullable(),
			'manufacturers'                 => Expect::array()->nullable(),
			'features'                      => Expect::array()->nullable(),
			'products'                      => Expect::array()->nullable(),
		]), $data);

		if ($data['dateFrom']) {
			$data['dateFrom'] = $data['dateFrom'] instanceof DateTime ? $data['dateFrom'] : new DateTime($data['dateFrom']);
		} else {
			$data['dateFrom'] = null;
		}

		if ($data['dateTo']) {
			$data['dateTo'] = $data['dateTo'] instanceof DateTime ? $data['dateTo'] : new DateTime($data['dateTo']);
		} else {
			$data['dateTo'] = null;
		}

		$dataObj = ArrayHash::from($data);
		$data    = [];

		$codesConfig = EshopSalesConfig::load('codes');

		$result = [];
		for ($i = 0; $i < $dataObj->bulkCreationCount; $i++) {

			if ($dataObj->bulkCreationCount > 1 || ($dataObj->autoGenerateCodeIfCodeIsEmpty)) {
				$dataObj->code = $this->generateCode();
			}

			if ($dataObj->editedId) {
				$sale = $this->get($dataObj->editedId);

				if (!$sale) {
					continue;
				}

				if ($dataObj->sites) {
					$siteIdent = Arrays::first((array) $dataObj->sites);
					if (!$siteIdent) {
						$sale->sites->clear();
					} else {
						if (!$sale->sites->toArray()) {
							foreach ($dataObj->sites as $siteIdent) {
								if ($site = $this->adminSites->get($siteIdent)) {
									$sale->addSite($site);
								}
							}
						} else {
							/** @var OrderSaleInSite $site */
							foreach ($sale->sites->toArray() as $site) {
								$site->site = $this->adminSites->get($siteIdent);
							}
						}
					}
				}
			} else {
				$sale = new OrderSale($dataObj->type, (float) $dataObj->amount, (float) $dataObj->fromPrice, $dataObj->maxRepetitions);
				foreach ($dataObj->sites as $siteIdent) {
					if ($site = $this->adminSites->get($siteIdent)) {
						$sale->addSite($site);
					}
				}
			}

			$sale->setType($dataObj->type);
			$sale->setDateFrom($dataObj->dateFrom);
			$sale->setDateTo($dataObj->dateTo);
			$sale->setAmount((float) $dataObj->amount);
			$sale->setFromPrice((float) $dataObj->fromPrice);
			$sale->setCustomerGroups((array) $dataObj->customerGroups);
			$sale->setCategories((array) $dataObj->categories);
			$sale->setManufacturers((array) $dataObj->manufacturers);
			$sale->setFeatures((array) $dataObj->features);
			$sale->setProducts((array) $dataObj->products);
			$sale->isActive       = $dataObj->isActive;
			$sale->code           = $dataObj->code;
			$sale->maxRepetitions = $dataObj->maxRepetitions;
			$sale->orderItem      = $dataObj->orderItem;
			$sale->orderGift      = $dataObj->orderGift;
			$sale->description    = $dataObj->description ?? null;

			$this->em->persist($sale)->flush();

			$result[(int) $sale->getId()] = $sale;
		}

		$this->eshopOrdersCache->getCache()->clean([
			Cache::Tags => ['orderSales'],
		]);

		return $result;
	}

	/**
	 * @param OrderSale[] $items
	 */
	public function sendDiscountCoupons(array $items): void
	{
		if (!EshopSalesConfig::load('autoGenerateDiscountCoupon.sendEmail.enable', true) || count($items) === 0) {
			return;
		}

		try {
			/** @var Order|null $order */
			$order = null;

			/** @var OrderSale[] $discountCoupons */
			$discountCoupons = [];

			foreach ($items as $item) {
				if ($item->orderItem && $item->orderItem->getProduct()->isDiscount) {
					$order = $item->orderItem->order;
					$discountCoupons[] = $item;
				}

				if ($item->orderGift && $item->orderGift->getProduct()->isDiscount) {
					$order = $item->orderGift->getOrder();
					$discountCoupons[] = $item;
				}
			}

			if (!$order || !$discountCoupons) {
				return;
			}

			$siteIdent = $order->site->getIdent();

			$template           = $this->mailTemplateFactory->create($siteIdent);
			$templateFile       = TEMPLATES_DIR . '/Front/default/EshopSales/discountCouponsEmail.latte';
			$originTemplateFile = __DIR__ . '/../../Model/_emails/discountCouponsEmail.latte';
			$template->setFile(file_exists($templateFile) ? $templateFile : $originTemplateFile);
			$template->orderSales = $items;
			$template->orderId    = $order->getId();
			$template->hasPhysicalDiscount = (bool) array_filter($discountCoupons, static function(OrderSale $os) {
				return ($os->orderItem && $os->orderItem->getProduct() && $os->orderItem->getProduct()->getMoreDataValue('discountShipmentMethod') === 'physical')
					|| ($os->orderGift && $os->orderGift->getProduct() && $os->orderGift->getProduct()->getMoreDataValue('discountShipmentMethod') === 'physical');
			});
			$template->orderId    = $order->getId();
			$template->setParameters([
				'originTemplate' => $originTemplateFile,
				'site'           => $order->site,
			]);

			$seller      = $this->sellers->getSellerForSite($siteIdent);
			$sellerEmail = Helpers::getSellerEmail($seller, $order->site, $order->lang);
			$sellerName  = Helpers::getSellerName($seller, $siteIdent, $order->lang);

			$message = new Message;
			$message->setFrom($sellerEmail, $sellerName);

			$bcc = EshopSalesConfig::load('autoGenerateDiscountCoupon.sendEmail.bcc');
			if ($bcc) {
				foreach ($bcc as $v) {
					$message->addBcc(trim($v));
				}
			}

			foreach ($discountCoupons as $discountCoupon) {
				$pdf = $this->discountCouponPdfBuilderFactory->create($discountCoupon);
				$message->addAttachment($pdf->getFileName(), $pdf->render(Destination::STRING_RETURN));
			}

			$addrInv = $order->getAddressInvoice();
			if (!$addrInv) {
				return;
			}

			$message->addTo(
				$addrInv->getEmail(),
				$addrInv->getFirstName() . ' ' . $addrInv->getLastName()
			);
			$message->setHtmlBody($template->renderToString());

			$this->mailer->send($message);
		} catch (Exception $e) {
			Debugger::log('Order sale email failed - ' . $order->getId(), 'orderSaleEmail');
			Debugger::log($e, 'orderSaleEmail');

			throw $e;
		}
	}

	public function bulkImportHolidays(): void
	{
		try {
			$config = EshopSalesConfig::load('holidaysCoupon');

			if (!$config['enable']) {
				return;
			}

			Debugger::log('OrderSales - bulkImportHolidays', 'scheduler/' . (new DateTime())->format('y-m-d'));
			$resourceContent = file_get_contents($config['resourceUrl']);
			if (!$resourceContent) {
				return;
			}

			$items = Json::decode($resourceContent, Json::FORCE_ARRAY);

			foreach ($items as $item) {
				$expiration = $config['expiration'];
				$name       = $item['name'];
				$currYear   = date('Y');
				$dateFrom   = DateTime::createFromFormat('d.m.Y', $item['date'] . $currYear);
				$dateTo     = null;

				if ($dateFrom) {
					$dateFrom->setTime(0, 0, 0);
					$dateTo = (($dateFrom)->modifyClone("+{$expiration} days"))->setTime(23, 59, 59);
				}

				$this->save([
					'type'                          => $config['type'],
					'dateFrom'                      => $dateFrom,
					'dateTo'                        => $dateTo,
					'amount'                        => (float) $config['amount'],
					'fromPrice'                     => (float) $config['fromPrice'],
					'isActive'                      => true,
					'code'                          => null,
					'maxRepetitions'                => null,
					'bulkCreationCount'             => 1,
					'orderItem'                     => null,
					'editedId'                      => null,
					'autoGenerateCodeIfCodeIsEmpty' => true,
					'description'                   => $name,
				]);
			}

			$this->eshopOrdersCache->getCache()->clean([
				Cache::Tags => ['orderSales'],
			]);
		} catch (Exception $e) {
			Debugger::log($e, 'orderSales-bulkImportHolidays');
		}
	}

	public function removeExpired(): void
	{
		try {
			if (!EshopSalesConfig::load('enableRemoveExpiredCoupons', false)) {
				return;
			}

			Debugger::log('OrderSales - removeExpired', 'scheduler/' . (new DateTime())->format('y-m-d'));

			$today = new DateTime;
			$today->setTime(0, 0, 0);
			$qb = $this->getEr()->createQueryBuilder('os');
			$qb->select('os.id')
				->where('os.dateTo IS NOT NULL AND os.dateTo < :today')
				->orderBy('os.dateTo')
				->setParameter('today', $today);

			$expiredItems = $qb->getQuery()->getArrayResult();

			if ($expiredItems) {
				$expiredItems = array_map(static fn($arr) => $arr['id'], $expiredItems);
				$qb           = $this->getEr()->createQueryBuilder('os');
				$qb->delete()
					->where($qb->expr()->in('os.id', $expiredItems))
					->getQuery()
					->execute();

				$this->eshopOrdersCache->getCache()->clean([
					Cache::Tags => ['orderSales'],
				]);
			}
		} catch (Exception $e) {
			Debugger::log($e, 'orderSales-removeExpired');
		}
	}

	/**
	 * @return OrderSale[]
	 */
	public function getDiscountsPurchasedInOrder(int $orderId): array
	{
		$qb = $this->getEr()->createQueryBuilder('os');

		$qb->join('os.orderItem', 'oi')
			->join('oi.order', 'o', 'WITH', 'o.id = :orderId')
			->setParameter('orderId', $orderId);

		return $qb->getQuery()->getResult();
	}

	public function getPurchasedDiscounts(): array
	{
		$result = [];
		/** @var OrderSale[] $salesFromOrderItems */
		$salesFromOrderItems = [];
		/** @var OrderSale[] $salesFromOrderGifts */
		$salesFromOrderGifts = [];
		/** @var UsedOrderSale[] $usedSalesFromOrderItems */
		$usedSalesFromOrderItems = [];
		/** @var UsedOrderSale[] $usedSalesFromOrderGifts */
		$usedSalesFromOrderGifts = [];
		$qb         = $this->em->getRepository(OrderItem::class)->createQueryBuilder('oi');
		/** @var OrderItem[] $orderItemDiscounts */
		$orderItemDiscounts = $qb->addSelect('p, o')
			->join('oi.order', 'o')
			->join('oi.product', 'p', 'WITH', 'p.isDiscount = 1')
			->addOrderBy('oi.order', 'desc')
			->getQuery()->getResult();

		if ($orderItemDiscounts) {
			$qb         = $this->getEr()->createQueryBuilder('os');
			$salesFromOrderItems = $qb->where($qb->expr()
				->in('os.orderItem', array_map(static fn(OrderItem $oi) => $oi->getId(), $orderItemDiscounts)))
				->getQuery()->getResult();

			$qb = $this->em->getRepository(UsedOrderSale::class)->createQueryBuilder('uos');
			$usedSalesFromOrderItems = $qb->where($qb->expr()
				->in('uos.createdByOrderItem', array_map(static fn(OrderItem $oi) => $oi->getId(), $orderItemDiscounts)))
				->getQuery()->getResult();
		}

		$qb = $this->em->getRepository(OrderGift::class)->createQueryBuilder('og');
		/** @var OrderGift[] $orderGiftDiscounts */
		$orderGiftDiscounts = $qb->addSelect('p, o')
							->join('og.order', 'o')
							->join('og.product', 'p', 'WITH', 'p.isDiscount = 1')
							->addOrderBy('og.order', 'desc')
							->getQuery()->getResult();

		if ($orderGiftDiscounts) {
			$qb         = $this->getEr()->createQueryBuilder('os');
			$salesFromOrderGifts = $qb->where($qb->expr()
			 	->in('os.orderGift', array_map(static fn(OrderGift $og) => $og->getId(), $orderGiftDiscounts)))
				->getQuery()->getResult();

			$qb = $this->em->getRepository(UsedOrderSale::class)->createQueryBuilder('uos');
			$usedSalesFromOrderGifts = $qb->where($qb->expr()
				->in('uos.createdByOrderGift', array_map(static fn(OrderGift $og) => $og->getId(), $orderGiftDiscounts)))
				->getQuery()->getResult();
		}

		// pro slev. poukazy vznikle z darku objednavky
		foreach ($orderGiftDiscounts as $discount) {
			$productName = $discount->getProduct()->getText()->name ?? '';
			$order = $discount->getOrder()->getId();
			$purchased = $discount->getOrder()->getCreatedTime();
			$invoice = $discount->getOrder()->getInvoice() ? $discount->getOrder()->getInvoice()->ident : null;

			$itemsNotApplied = [];
			foreach ($salesFromOrderGifts as $orderSale) {
				if ($orderSale->orderGift->getId() === $discount->getId()) {
					$itemsNotApplied[] = [
						'status'                => 'notApplied',
						'code'                  => $orderSale->code,
						'dateFrom'              => $orderSale->getDateFrom() ? $orderSale->getDateFrom()->format('d.m.Y') : null,
						'dateTo'                => $orderSale->getDateTo() ? $orderSale->getDateTo()->format('d.m.Y') : null,
						'appliedInOrderInvoice' => null,
						'appliedInOrder'        => null,
					];
				}
			}

			$itemsApplied = [];
			foreach ($usedSalesFromOrderGifts as $usedOrderSale) {
				if ($usedOrderSale->createdByOrderGift->getId() === $discount->getId() && $usedOrderSale->order) {
					$itemsApplied[] = [
						'status'                => 'applied',
						'code'                  => $usedOrderSale->code,
						'dateFrom'              => $usedOrderSale->dateFrom ? $usedOrderSale->dateFrom->format('d.m.Y') : null,
						'dateTo'                => $usedOrderSale->dateTo ? $usedOrderSale->dateTo->format('d.m.Y') : null,
						'appliedInOrderInvoice' => $usedOrderSale->order->getInvoice() ? $usedOrderSale->order->getInvoice()->ident : null,
						'appliedInOrder'        => $usedOrderSale->order->getId()
					];
				}
			}

			// vygenerovane nebo aplikovane
			foreach ([$itemsNotApplied, $itemsApplied] as $arr) {
				foreach ($arr as &$item) {
					$item['productName'] = $productName;
					$item['order'] = $order;
					$item['purchased'] = $purchased;
					$item['invoice'] = $invoice;
					$result[] = $item;
				}
			}

			// nevygenerovane
			$countNotCreated = max(0, $discount->getQuantity() - (count($itemsNotApplied) + count($itemsApplied)));
			for ($i = 0; $i < $countNotCreated; $i++) {
				$result[] = [
					'status'                => !$discount->getOrder()->isPaid ? 'codeNotExists' : 'applied',
					'productName'           => $productName,
					'order'                 => $order,
					'purchased'             => $purchased,
					'invoice'               => $invoice,
					'code'                  => null,
					'dateFrom'              => null,
					'dateTo'                => null,
					'appliedInOrderInvoice' => null,
					'appliedInOrder'        => null
				];
			}
		}

		// pro slev. poukazy vznikle z polozky objednavky
		foreach ($orderItemDiscounts as $discount) {
			$productName = $discount->getProduct()->getText()->name ?? '';
			$order = $discount->getOrder()->getId();
			$purchased = $discount->order->getCreatedTime();
			$invoice = $discount->order->getInvoice() ? $discount->order->getInvoice()->ident : null;

			$itemsNotApplied = [];
			foreach ($salesFromOrderItems as $orderSale) {
				if ($orderSale->orderItem->getId() === $discount->getId()) {
					$itemsNotApplied[] = [
						'status'                => 'notApplied',
						'code'                  => $orderSale->code,
						'dateFrom'              => $orderSale->getDateFrom() ? $orderSale->getDateFrom()->format('d.m.Y') : null,
						'dateTo'                => $orderSale->getDateTo() ? $orderSale->getDateTo()->format('d.m.Y') : null,
						'appliedInOrderInvoice' => null,
						'appliedInOrder'        => null,
					];
				}
			}

			$itemsApplied = [];
			foreach ($usedSalesFromOrderItems as $usedOrderSale) {
				if ($usedOrderSale->createdByOrderItem->getId() === $discount->getId() && $usedOrderSale->order) {
					$itemsApplied[] = [
						'status'                => 'applied',
						'code'                  => $usedOrderSale->code,
						'dateFrom'              => $usedOrderSale->dateFrom ? $usedOrderSale->dateFrom->format('d.m.Y') : null,
						'dateTo'                => $usedOrderSale->dateTo ? $usedOrderSale->dateTo->format('d.m.Y') : null,
						'appliedInOrderInvoice' => $usedOrderSale->order->getInvoice() ? $usedOrderSale->order->getInvoice()->ident : null,
						'appliedInOrder'        => $usedOrderSale->order->getId()
					];
				}
			}

			// vygenerovane nebo aplikovane
			foreach ([$itemsNotApplied, $itemsApplied] as $arr) {
				foreach ($arr as &$item) {
					$item['productName'] = $productName;
					$item['order'] = $order;
					$item['purchased'] = $purchased;
					$item['invoice'] = $invoice;
					$result[] = $item;
				}
			}

			// nevygenerovane
			$countNotCreated = max(0, $discount->getQuantity() - (count($itemsNotApplied) + count($itemsApplied)));
			for ($i = 0; $i < $countNotCreated; $i++) {
				$result[] = [
					'status'                => !$discount->getOrder()->isPaid ? 'codeNotExists' : 'applied',
					'productName'           => $productName,
					'order'                 => $order,
					'purchased'             => $purchased,
					'invoice'               => $invoice,
					'code'                  => null,
					'dateFrom'              => null,
					'dateTo'                => null,
					'appliedInOrderInvoice' => null,
					'appliedInOrder'        => null
				];
			}
		}

		return $result;
	}

	public function generateCode(): string
	{
		$codesConfig = EshopSalesConfig::load('codes');

		return Random::generate($codesConfig['length'], $codesConfig['charlist']);
	}

	public function getCodes(): array
	{
		return array_values($this->getEr()->findPairs([], 'code', ['code'], 'id'));
	}

	/**
	 * @param int|string $id
	 */
	public function remove($id): bool
	{
		$result = parent::remove($id);

		if ($result) {
			$this->eshopOrdersCache->getCache()->clean([
				Cache::Tags => ['orderSales'],
			]);
		}

		return $result;
	}

}
