<?php declare(strict_types = 1);

namespace EshopSales\FrontModule\Components\Order;

use Core\Model\Event\Event;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\TextInput;
use EshopOrders\FrontModule\Model\CartHelper;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Customers;
use EshopSales\FrontModule\Model\OrderSales;
use EshopSales\Model\Config;
use EshopSales\Model\Entities\OrderSale;
use EshopSales\Model\EshopSalesConfig;
use Exception;
use Nette\Http\SessionSection;
use Nette\Utils\ArrayHash;
use Users\Model\Entities\User;
use Users\Model\Http\UserStorage;

class DiscountForm extends BaseControl
{
	private Carts            $cartsService;
	protected CartHelper     $cartHelperService;
	protected OrderSales     $orderSales;
	protected SessionSection $sessionSection;
	protected Config         $config;
	protected UserStorage    $userStorage;
	protected Customers      $customersService;

	public function __construct(
		OrderSales  $orderSales,
		Config      $config,
		Carts       $carts,
		CartHelper  $cartHelper,
		UserStorage $userStorage,
		Customers   $customersService
	)
	{
		$this->orderSales = $orderSales;
		$this->config     = $config;

		$this->cartsService      = $carts;
		$this->cartHelperService = $cartHelper;
		$this->userStorage       = $userStorage;
		$this->customersService  = $customersService;
	}

	/**
	 * @return BaseForm
	 */
	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax()
			->setShowLangSwitcher(false);

		$form->addText('discountCode', 'eshopOrders.orderPage.discountCode')
			->setRequired();

		$form->addSubmit('submit', 'eshopSalesFront.discountForm.submit', 'darkest-grey-bg-color dark-grey-bg-hover-color');

		$form->onValidate[] = [$this, 'formOnValidate'];
		$form->onSuccess[]  = [$this, 'formOnSuccess'];

		return $form;
	}

	/**
	 * @param BaseForm  $form
	 * @param ArrayHash $values
	 *
	 * @throws Exception
	 */
	public function formOnValidate(BaseForm $form, ArrayHash $values): void
	{
		$cart = $this->cartsService->getCurrentCart();

		/** @var User $user */
		$user     = $this->userStorage->getIdentity();
		$customer = $this->userStorage->getIdentity() ? $this->customersService->getByUser($user) : null;

		/** @var TextInput $discountCodeControl */
		$discountCodeControl = $form['discountCode'];

		$cartItemsPrice          = $cart->getCartItemsPrice();
		$cartSales               = $this->orderSales->getOrderSalesFromCart($cartItemsPrice);
		$cartItemsPriceAfterSale = $cartItemsPrice - abs($cart->calculateDiscounts());

		$formSale = $this->orderSales->isValidDiscountCode($values->discountCode, $cartItemsPriceAfterSale);

		if ($formSale) {
			$productsCanUse = false;
			foreach ($cart->getCartItems() as $item) {
				if (
					!$item->product
					|| $formSale->canUseOnProduct($item->product, $customer)
				) {
					$productsCanUse = true;
					break;
				}
			}

			if (!$productsCanUse) {
				$discountCodeControl->addError('eshopSalesFront.discountForm.productsForCodeNotFound');
			}

			if (!EshopSalesConfig::load('front.allowForGroupCustomers', true)) {
				if ($customer && $customer->getGroupCustomers()
					&& (empty($formSale->customerGroups) && !in_array($customer->getGroupCustomers()->getId(), $formSale->customerGroups))) {
					$discountCodeControl->addError('eshopSalesFront.discountForm.notAllowedForGroupCustomer');
				}
			}

			if ($formSale->type === OrderSale::TYPE_PERCENT) {
				foreach ($cartSales as $sale) {
					if ($sale->type === OrderSale::TYPE_PERCENT) {
						$discountCodeControl->addError('eshopSalesFront.discountForm.onlyOnePercentCode');
						break;
					}
				}
			} else if ($formSale->type === OrderSale::TYPE_DELIVERY_PRICE_FIRST_ORDER) {
				if (!$this->orderSales->canUseOnFirstOrder()) {
					$discountCodeControl->addError('eshopSalesFront.discountForm.canUseOnlyForFirstOrder');
				}
			}

			if (count($cartSales) === $this->config->get('discountCouponsCanBeRedeemedAtOnce')) {
				$discountCodeControl->addError('eshopSalesFront.discountForm.discountCodesExhausted');
			}
		} else {
			$discountCodeControl->addError('eshopSalesFront.discountForm.discountCodeInvalid');
		}

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	/**
	 * @param BaseForm  $form
	 * @param ArrayHash $values
	 */
	public function formOnSuccess(BaseForm $form, ArrayHash $values): void
	{
		$orderSale = $this->orderSales->getByCode($values->discountCode);

		if ($orderSale === null) {
			return;
		}

		$this->orderSales->addOrderSaleToCart($orderSale->id);
		$this->eventDispatcher->dispatch(new Event(), 'eshopSales.discountCodesChanged');
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}
}

