<?php declare(strict_types = 1);

namespace EshopSales\FrontModule\Model;

use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Sites;
use Currency\Model\Config;
use Currency\Model\Currencies;
use Currency\Model\Exchange;
use Doctrine\ORM\Query;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\Dao\Cart;
use EshopOrders\FrontModule\Model\Dao\Discount;
use EshopOrders\FrontModule\Model\Event\FillDaoItemsEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use EshopSales\FrontModule\Model\Event\OrderSalesEvent;
use EshopSales\FrontModule\Model\Event\OrderSalesPriceEvent;
use EshopSales\Model\Entities\OrderSale;
use Exception;
use Nette\Caching\Cache;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Utils\DateTime;
use Users\Model\Security\User;

class OrderSales extends BaseFrontEntityService
{
	/** @var array */
	protected $cAutoSales;

	/** @var string */
	protected $entityClass = OrderSale::class;

	/** @var string */
	private const SESSION_SECTION = 'appliedDiscountCouponsSection';

	/** @var string */
	private const KEY_CODES = 'codes';

	protected SessionSection   $sessionSection;
	protected EventDispatcher  $eventDispatcher;
	protected ProductsFacade   $productsFacade;
	protected Exchange         $exchange;
	protected Currencies       $currencies;
	protected Sites            $sites;
	protected User             $user;
	protected Customers        $customers;
	protected EshopOrdersCache $eshopOrdersCache;

	protected array $cDao = [];

	public function __construct(
		Session          $session,
		EventDispatcher  $eventDispatcher,
		Exchange         $exchange,
		ProductsFacade   $productsFacade,
		Sites            $sites,
		User             $user,
		Customers        $customers,
		Currencies       $currencies,
		EshopOrdersCache $eshopOrdersCache
	)
	{
		$this->sessionSection   = $session->getSection(self::SESSION_SECTION);
		$this->eventDispatcher  = $eventDispatcher;
		$this->exchange         = $exchange;
		$this->productsFacade   = $productsFacade;
		$this->sites            = $sites;
		$this->user             = $user;
		$this->customers        = $customers;
		$this->currencies       = $currencies;
		$this->eshopOrdersCache = $eshopOrdersCache;
	}

	/**
	 * @return Dao\OrderSale[]
	 * @throws Exception
	 */
	public function getAutoSales(): array
	{
		if ($this->cAutoSales === null) {
			$today  = new DateTime();
			$result = $this->getEr()->createQueryBuilder('os')
				->where('os.code IS NULL')->andWhere('os.isActive = 1')
				->andWhere('os.dateFrom >= :today OR os.dateFrom IS NULL')->andWhere('os.dateTo <= :today OR os.dateTo IS NULL')
				->setParameter('today', $today)
				->orderBy('os.fromPrice', 'ASC')->getQuery()->getArrayResult();

			$this->cAutoSales = $result ? $this->fillDao($result) : [];
		}

		return $this->cAutoSales;
	}

	public function getNextSales(float $cartItemsPrice): array
	{
		$result    = [];
		$autoSales = $this->getAutoSales();

		foreach ($autoSales as $k => $sale) {
			if ($cartItemsPrice >= $sale->fromPrice)
				continue;

			$result[] = $sale;
		}

		return $result;
	}

	/**
	 * @param array $codes
	 *
	 * @return QueryBuilder
	 * @throws Exception
	 */
	private function getQueryBuilderByValid(array $codes, string $column = 'code', bool $ignoreSite = false): QueryBuilder
	{
		$now = (new DateTime())->format('Y-m-d 00:00:00');

		$qb = $this->getEr()->createQueryBuilder('os');
		$qb->leftJoin('os.sites', 'sites')
			->leftJoin('sites.site', 'site')
			->andWhere('os.isActive = 1')
			->andWhere('os.dateFrom <= :now OR os.dateFrom IS NULL')
			->andWhere('os.dateTo >= :now OR os.dateTo IS NULL')
			->andWhere('os.' . $column . ' IN (:codes) OR os.code IS NULL')
			->setParameters([
				'now'   => $now,
				'codes' => $codes,
			]);
		if (!$ignoreSite) {
			$qb->andWhere('site.ident = :site OR site.ident IS NULL')
				->setParameter('site', $this->sites->getCurrentSite()->getIdent());
		}

		return $qb;
	}

	public function isValidDiscountCode(string $code, float $cartItemsPrice, bool $ignoreSite = false): ?Dao\OrderSale
	{
		$result = $this->getQueryBuilderByValid([$code], 'code', $ignoreSite)
			->andWhere('os.code IS NOT NULL')
			->getQuery()
			->setMaxResults(1)
			->getOneOrNullResult(Query::HYDRATE_ARRAY);

		if ($result) {
			$result = $this->fillDao([$result]);
			$result = $this->validateCartPrice($result, $cartItemsPrice);
		} else {
			return null;
		}

		$exist = isset(array_values($result)[0]);

		return $exist ? array_values($result)[0] : null;
	}

	/**
	 * @param int $id
	 */
	public function addOrderSaleToCart(int $id): void
	{
		$arr                                     = $this->sessionSection->{self::KEY_CODES};
		$arr[]                                   = $id;
		$arr                                     = array_unique($arr); // Removes duplicate values
		$this->sessionSection->{self::KEY_CODES} = $arr;
	}

	/**
	 * @param Dao\OrderSale[] $orderSales
	 */
	private function rewriteCodesInCart(array $orderSales): void
	{
		$this->sessionSection->{self::KEY_CODES} = array_map(static function(Dao\OrderSale $sale) {
			return $sale->id;
		}, $orderSales);
	}

	public function clearCodesInCart(): void
	{
		$this->rewriteCodesInCart([]);
	}

	/**
	 * @param int $id
	 */
	public function removeFromCart(int $id): void
	{
		foreach ($this->sessionSection->{self::KEY_CODES} as $key => $tmpId) {
			if ($tmpId === $id) {
				unset($this->sessionSection->{self::KEY_CODES}[$key]);
				break;
			}
		}
	}

	public function removeFromCartByCode(string $code): void
	{
		$sale = $this->getByCode($code);

		if ($sale) {
			$this->removeFromCart($sale->id);
		}
	}

	/**
	 * @param float $cartItemsPrice
	 *
	 * @return Dao\OrderSale[]
	 * @throws Exception
	 */
	public function getOrderSalesFromCart(float $cartItemsPrice): array
	{
		$now       = (new DateTime())->format('Y-m-d-00-00-00');
		$siteIdent = $this->sites->getCurrentSite()->getIdent();
		$storedIds = $this->sessionSection->{self::KEY_CODES} ?? [];

		$key = 'orderSalesFromCart_' . $siteIdent . '_' . $now . '_' . serialize($storedIds);
		/** @var ?array $result */
		$result = $this->eshopOrdersCache->getCache()->load($key, function(&$dep) use ($storedIds) {
			$dep = [
				Cache::Tags       => ['orderSales', 'orderSalesFromCart'],
				Cache::EXPIRATION => '1 day',
			];

			$this->eshopOrdersCache->getCache()->clean([
				Cache::Tags => ['orderSalesFromCart'],
			]);

			return $this->getQueryBuilderByValid($storedIds, 'id')
				->addOrderBy('os.code')
				->getQuery()
				->getArrayResult();
		});

		if ($result) {
			$result = $this->fillDao($result);
			$result = $this->validateCartPrice($result, $cartItemsPrice);
		}

		$withNullCode = $result
			? array_filter($result, static function(Dao\OrderSale $orderSale): bool {
				return $orderSale->code === null;
			})
			: [];

		$withNotNullCode = $result
			? array_filter($result, static function(Dao\OrderSale $orderSale): bool {
				return $orderSale->code !== null;
			})
			: [];

		if ($withNullCode) {
			usort($withNullCode, fn(Dao\OrderSale $a, Dao\OrderSale $b) => $a->getSortPosition() <=> $b->getSortPosition());
			$withNullCode = [end($withNullCode)];
		}

		$orderSales = array_merge($withNotNullCode, $withNullCode);

		if ($orderSales) {
			usort($orderSales, fn(Dao\OrderSale $a, Dao\OrderSale $b) => $a->getSortPosition() <=> $b->getSortPosition());
		}

		// rewrite old codes. The original codes may no longer be valid
		$this->rewriteCodesInCart($orderSales);

		return $orderSales;
	}

	public function get(int $id): ?Dao\OrderSale
	{
		$result = $this->getEr()->createQueryBuilder('os')
			->where('os.id = :id')->setParameter('id', $id)
			->getQuery()->setMaxResults(1)->getOneOrNullResult(Query::HYDRATE_ARRAY);

		return $result ? $this->fillDao([$result])[$id] : null;
	}

	public function getByCode(string $code): ?Dao\OrderSale
	{
		if ($code == OrderSale::AUTO_SALE_ID)
			$code = null;

		$result = $this->getEr()->createQueryBuilder('os')
			->where('os.code = :code')->setParameter('code', $code)
			->getQuery()->setMaxResults(1)->getOneOrNullResult(Query::HYDRATE_ARRAY);

		return $result ? $this->fillDao([$result])[$result['id']] : null;
	}

	/**
	 * @return array<string, Dao\OrderSale>
	 */
	public function getByCodes(array $codes): array
	{
		$result = [];

		return $this->eshopOrdersCache->getCache()->load('orderSalesByCodes_' . md5(implode('_', $codes)), function(&$dep) use ($codes, &$result) {
			$dep = [
				Cache::Tags       => ['orderSales'],
				Cache::EXPIRATION => '10 minutes',
			];

			foreach ($this->getEr()->createQueryBuilder('os')
				         ->where('os.code IN (:codes)')->setParameter('codes', $codes)
				         ->getQuery()->getArrayResult() as $row) {
				$result[$row['code']] = $this->fillDao([$row])[$row['id']];
			}

			return $result;
		});
	}

	public function setAutoSalesToCart(Cart $cart): void
	{
		$cartItemsPrice = $cart->getCartItemsPrice();
		$sales          = $this->getOrderSalesFromCart($cartItemsPrice);

		$cart->removeDiscount(OrderSale::AUTO_SALE_ID);
		$productsPrice = array_sum(array_map(static fn($v) => $v->getTotalPrice(), $cart->getProductsForDiscount()));
		$customer      = $this->user->isLoggedIn() ? $this->customers->getOrCreateCustomer($this->user->getId()) : null;
		$currency      = $this->currencies->getCurrent();

		if ($sales) {
			usort($sales, static fn(Dao\OrderSale $a, Dao\OrderSale $b) => $a->getSortPosition() <=> $b->getSortPosition());
		}

		$discounts = [];

		foreach ($sales as $k => $sale) {
			$code = $sale->isAutoSale() ? OrderSale::AUTO_SALE_ID : $sale->code;

			if ($sale->type === OrderSale::TYPE_DELIVERY_PRICE) {
				$discount                            = new Discount($code, $sale->value, $sale->type, 'eshopSales');
				$discount->amountInBaseCurrency      = $sale->valueInBaseCurrency;
				$discount->calculateDiscountCallback = static fn(Discount $discount) => 0;
				$discounts[$code]                    = $discount;

				$discount->text = $this->translator->translate('eshopSalesFront.discountsGrid.speditionSale') . ' - ' . $discount->id;
				$discount->description = $sale->description;

				if ($cart->spedition) {
					$cart->spedition->priceInBaseCurrency = $sale->valueInBaseCurrency;
					$cart->spedition->setPrice($sale->value);
				}
			} else if ($sale->type === OrderSale::TYPE_DELIVERY_PRICE_FIRST_ORDER) {
				$discount                            = new Discount($code, $sale->value, $sale->type, 'eshopSales');
				$discount->amountInBaseCurrency      = $sale->valueInBaseCurrency;
				$discount->calculateDiscountCallback = static fn(Discount $discount) => 0;
				$discounts[$code]                    = $discount;

				$discount->text = $this->translator->translate('eshopSalesFront.discountsGrid.speditionSaleFirstOrder') . ' - ' . $discount->id;
				$discount->description = $sale->description;

				if ($cart->spedition) {
					$cart->spedition->priceInBaseCurrency = $sale->valueInBaseCurrency;
					$cart->spedition->setPrice($sale->value);
				}
			} else if ($sale->type === OrderSale::TYPE_RANDOM_CAT_PRODUCT) {
				$discount                            = new Discount($code, $sale->value, $sale->type, 'eshopSales');
				$discount->calculateDiscountCallback = static function(Discount $discount) {
					return 0;
				};
				$discounts[$code]                    = $discount;

				$discount->text = $this->translator->translate('eshopSalesFront.discountsGrid.freeProduct') . ' - ' . $discount->id;
				$discount->description = $sale->description;
			} else if ($sale->type === OrderSale::TYPE_PRODUCT) {
				$product = $this->productsFacade->getProduct((int) $sale->value);
				if ($product) {
					$discount                            = new Discount($code, $sale->value, $sale->type, 'eshopSales');
					$discount->title                     = $product->name;
					$discount->calculateDiscountCallback = static function(Discount $discount) {
						return 0;
					};
					$discounts[$code]                    = $discount;

					$discount->text  = $this->translator->translate('eshopSalesFront.discountsGrid.freeProduct') . ' - ' . $product->name;
					$discount->title = $discount->text;
					$discount->description = $sale->description;
				}
			} else {
				$discount                       = new Discount($code, $sale->value, $sale->type, 'eshopSales');
				$discount->amountInBaseCurrency = $sale->valueInBaseCurrency;
				$discount->typeSymbol           = OrderSale::TYPES[$sale->type]['symbol'] ?: ($currency ? $currency->symbol : '');

				if ($productsPrice <= 0) {
					$discount->discount               = 0;
					$discount->discountInBaseCurrency = 0;
				} else {
					foreach ($cart->getProductsForDiscount() as $cartItem) {
						if ($cartItem->product && !$sale->canUseOnProduct($cartItem->product, $customer)) {
							continue;
						}

						if ($sale->type === OrderDiscount::TYPE_PERCENT) {
							$discount->discount -= $cartItem->getTotalPrice() * ($sale->value / 100);
						} else {
							$discount->discount = -$sale->value;

							if ($productsPrice + $discount->discount < 0) {
								$discount->discount -= ($productsPrice + $discount->discount);
							}

							break;
						}

						$discount->discount = round($discount->discount, $currency->decimals);
					}

					$productsPrice                    += $discount->discount;
					$discount->discountInBaseCurrency = $discount->discount;
				}

				$discounts[$code] = $discount;

				$discount->text = $this->translator->translate('eshopOrdersFront.orderPage.yourDiscount', null, [
					'c' => $discount->id !== 'ESHOPSALESAUTO' ? $discount->id : $sale->getAutoSaleCode(),
					'p' => $discount->amount . ' ' . $discount->typeSymbol,
				]);
				$discount->description = $sale->description;
			}
		}

		$this->eventDispatcher->dispatch(new FillDaoItemsEvent($discounts), 'eshopOrders.cartSetDiscounts');

		foreach ($discounts as $k => $discount) {
			$cart->addDiscount((string) $k, $discount);
		}
	}

	/**
	 * @param Dao\OrderSale[] $orderSales
	 * @param float           $cartPrice
	 *
	 * @return array
	 */
	protected function validateCartPrice(array $orderSales, float $cartPrice): array
	{
		$this->eventDispatcher->dispatch(new OrderSalesPriceEvent($orderSales, $cartPrice), OrderSales::class . '::validatePriceFrom');

		return $orderSales;
	}

	protected function fillDao(array $orderSales): array
	{
		$result = [];
		$loaded = [];

		foreach ($orderSales as $orderSale) {
			if (isset($this->cDao[$orderSale['id']])) {
				$result[(int) $orderSale['id']] = $this->cDao[$orderSale['id']];
				continue;
			}

			$dao                          = new Dao\OrderSale();
			$dao->id                      = (int) $orderSale['id'];
			$dao->code                    = $orderSale['code'];
			$dao->type                    = $orderSale['type'];
			$dao->description             = $orderSale['description'];
			$dao->value                   = (float) $orderSale['amount'];
			$dao->valueInBaseCurrency     = (float) $orderSale['amount'];
			$dao->fromPriceInBaseCurrency = (float) $orderSale['fromPrice'];
			$dao->fromPrice               = $this->exchange->change((float) $orderSale['fromPrice']);
			$dao->customerGroups          = $orderSale['customerGroups'] ? array_map(fn($v) => (int) $v, explode(',', (string) $orderSale['customerGroups'])) : [];
			$dao->categories              = $orderSale['categories'] ? array_map(fn($v) => (int) $v, explode(',', (string) $orderSale['categories'])) : [];
			$dao->manufacturers           = $orderSale['manufacturers'] ? array_map(fn($v) => (int) $v, explode(',', (string) $orderSale['manufacturers'])) : [];
			$dao->features                = $orderSale['features'] ? array_map(fn($v) => (int) $v, explode(',', (string) $orderSale['features'])) : [];
			$dao->products                = $orderSale['products'] ? array_map(fn($v) => (int) $v, explode(',', (string) $orderSale['products'])) : [];
			$dao->dateFrom                = $orderSale['dateFrom'];
			$dao->dateTo                  = $orderSale['dateTo'];

			if ($dao->type !== OrderSale::TYPE_PERCENT && $orderSale['amount']) {
				$dao->value = $this->exchange->change((float) $orderSale['amount']);
			}

			if (Config::load('roundOrderSales')) {
				$dao->value     = round($dao->value);
				$dao->fromPrice = round($dao->fromPrice);
			}

			$loaded[$dao->id] = $dao;
		}

		if ($loaded) {
			$this->eventDispatcher->dispatch(new OrderSalesEvent($loaded), OrderSales::class . '::fillDao');

			foreach ($loaded as $id => $dao) {
				$result[$id]     = $dao;
				$this->cDao[$id] = $dao;
			}
		}

		return $result;
	}

	public function canUseOnFirstOrder(
		?string $email = null,
		?string $phone = null
	): bool
	{
		$user     = $this->user->getId();
		$customer = $user ? $this->customers->getByUser($user) : null;
		if ($customer) {
			return $customer->getOrders()->count() === 0;
		} else {
			if ($email) {
				$count = $this->em->getRepository(Order::class)->createQueryBuilder('o')
					->select('COUNT(o.id) as count')
					->innerJoin('o.addressInvoice', 'ai', Join::WITH, 'ai.email = :email')
					->setParameter('email', $email)
					->getQuery()->getOneOrNullResult()['count'] ?? 0;

				if ($count > 0) {
					return false;
				}
			}

			if ($phone) {
				$count = $this->em->getRepository(Order::class)->createQueryBuilder('o')
					->select('COUNT(o.id) as count')
					->innerJoin('o.addressInvoice', 'ai', Join::WITH, 'ai.phone = :phone')
					->setParameter('phone', $phone)
					->getQuery()->getOneOrNullResult()['count'] ?? 0;

				if ($count > 0) {
					return false;
				}
			}
		}

		return true;
	}

	public function loadProductTagSalePrice(Product $product): void
	{
		if ($product->discountDisabled) {
			return;
		}

		$sales         = [];
		$salesFromCart = $this->getOrderSalesFromCart($product->getPrice());

		foreach ($salesFromCart as $sale) {
			if ($sale->isAutoSale()) {
				$sales[] = $sale;
			}
		}

		if (!empty($product->tags)) {
			foreach ($this->getByCodes(array_keys($product->tags)) as $sale) {
				$sales[] = $sale;
			}
		}

		if (empty($sales)) {
			return;
		}

		$maxDiscount    = 0;
		$maxDiscountDto = null;
		$customer       = $this->user->isLoggedIn() ? $this->customers->get($this->user->getId()) : null;
		foreach ($sales as $sale) {
			if (!$sale->canUseOnProduct($product, $customer)) {
				continue;
			}

			if ($sale->type === OrderDiscount::TYPE_PERCENT) {
				$discount = $product->getPrice() * ($sale->valueInBaseCurrency / 100);
			} else {
				$discount = abs($sale->valueInBaseCurrency);
			}

			if ($discount > $maxDiscount) {
				$maxDiscount    = $discount;
				$maxDiscountDto = $sale;
			}
		}

		if ($maxDiscount > 0) {
			$currency        = $this->currencies->getCurrent();
			$defaultCurrency = $this->currencies->getAll()[$this->currencies->getDefaultCode()] ?? null;
			$decimals        = $currency->decimals ?? ($defaultCurrency ? (int) $defaultCurrency->decimals : 0);

			$product->moreData['tagSalesPrice'] = [
				'price'    => $product->getPrice() - round($maxDiscount, $decimals),
				'discount' => round($maxDiscount, $decimals),
				'code'     => $maxDiscountDto->code,
				'dto'      => $maxDiscountDto,
			];
		}
	}
}
