<?php declare(strict_types = 1);

namespace EshopSales\FrontModule\Model\Subscribers;

use Core\Model\Event\Event;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopSales\FrontModule\Model\Dao\OrderSale;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Event\FillDaoEvent;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use EshopSales\FrontModule\Model\OrderSales;
use Exception;

class CartSubscriber implements EventSubscriberInterface
{
	protected OrderSales             $orderSales;
	protected Carts                  $carts;
	protected EntityManagerDecorator $entityManager;

	public function __construct(
		OrderSales             $orderSales,
		Carts                  $carts,
		EntityManagerDecorator $entityManager
	)
	{
		$this->orderSales    = $orderSales;
		$this->carts         = $carts;
		$this->entityManager = $entityManager;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.cartFillDao'              => ['cartFillDao', 99],
			'eshopOrders.cartUpdateItem'           => ['cartUpdateItem', 99],
			'eshopOrders.orderBeforeSave'          => ['orderBeforeSave', 99],
			'eshopCheckout.orderBeforeSave'        => ['orderCheckoutBeforeSave', 99],
			'eshopSales.discountCodesChanged'      => ['discountCodesChanged', 100],
			'eshopOrders.admin.addDiscountToOrder' => ['addDiscountToOrder', 100],
		];
	}

	public function discountCodesChanged(Event $event): void
	{
		$this->orderSales->setAutoSalesToCart($this->carts->getCurrentCart());
	}

	/**
	 * @param FillDaoEvent $event
	 *
	 * @throws Exception
	 */
	public function cartFillDao(FillDaoEvent $event): void
	{
		$cart = $event->cart;

		$this->orderSales->setAutoSalesToCart($cart);
	}

	/**
	 * @param UpdatedCartItemEvent $event
	 */
	public function cartUpdateItem(UpdatedCartItemEvent $event): void
	{
		$cartItemsCount = $this->carts->getCurrentCart()->getItemsCount();

		if ($cartItemsCount === 0) {
			$this->orderSales->clearCodesInCart();
		}
	}

	/**
	 * @param OrderEvent $event
	 *
	 * @throws Exception
	 */
	public function orderBeforeSave(OrderEvent $event): void
	{
		$currentCartItemsPrice = $this->carts->getCurrentCart()->getCartItemsPrice();
		$sales                 = $this->orderSales->getOrderSalesFromCart($currentCartItemsPrice);

		/** @var \EshopSales\Model\Entities\OrderSale[] $salesTmp */
		$salesTmp = array_map(fn(OrderSale $orderSale) => $this->orderSales->getEr()->find($orderSale->id), $sales);

		$this->processOrderSale($event->order, $salesTmp);
		$this->orderSales->clearCodesInCart();
	}

	public function orderCheckoutBeforeSave(OrderEvent $event): void
	{
		/** @var \EshopSales\Model\Entities\OrderSale[] $sales */
		$sales = array_map(
			fn(OrderDiscount $orderDiscount) => $this->orderSales->getEr()->findOneBy(['code' => $orderDiscount->getCode()]),
			$event->order->getOrderDiscounts()->toArray()
		);

		$this->processOrderSale($event->order, $sales);
	}

	public function addDiscountToOrder(OrderEvent $event): void
	{
		/** @var \EshopSales\Model\Entities\OrderSale[] $sales */
		$sales = array_map(
			fn(OrderDiscount $orderDiscount) => $this->orderSales->getEr()->findOneBy(['code' => $orderDiscount->getCode()]),
			$event->order->getOrderDiscounts()->toArray()
		);

		$this->processOrderSale($event->order, $sales);
	}

	/**
	 * @param \EshopSales\Model\Entities\OrderSale[]|null[] $sales
	 */
	private function processOrderSale(Order $order, array $sales): void
	{
		foreach ($sales as $sale) {
			if ($sale === null) {
				continue;
			}

			if (!empty($sale->code)) {
				$this->entityManager->persist($sale->toUsedOrderSale($order, $sale));
			}
			$sale->decreaseCurrentRepetitions();

			if ($sale->currentRepetitions === 0) {
				$this->entityManager->remove($sale);
			} else {
				$this->entityManager->persist($sale);
			}
		}
	}

}
