<?php declare(strict_types = 1);

namespace EshopSales\FrontModule\Model\Subscribers;

use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\Event\ControlEvent;
use Core\Model\UI\FrontPresenter;
use EshopCatalog\FrontModule\Components\ProductPreview;
use EshopCatalog\FrontModule\Presenters\DefaultPresenter;
use EshopOrders\FrontModule\Model\Carts;
use EshopSales\FrontModule\Model\OrderSales;
use EshopSales\Model\Config;
use EshopSales\Model\EshopSalesConfig;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class FrontSubscriber implements EventSubscriberInterface
{
	protected Config     $config;
	protected OrderSales $orderSales;
	protected Carts      $cartService;

	public function __construct(
		Config     $config,
		OrderSales $orderSales,
		Carts      $carts
	)
	{
		$this->config      = $config;
		$this->orderSales  = $orderSales;
		$this->cartService = $carts;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			FrontPresenter::class . '::beforeRender'    => ['beforeRender', 100],
			'eshopOrders.orderCartDetail.renderSummary' => ['orderCartDetailRenderSummary', 100],
			'eshopOrders.orderCartDetail.render'        => ['orderCartDetailRender', 100],
			'eshopOrders.cartDetail.render'             => ['cartDetailRender', 100],
			'eshopOrders.orderSummary.render'           => ['orderSummaryRender', 100],
			'eshopCatalog.productPreview.render'        => ['productPreviewRender', 100],
		];
	}

	public function productPreviewRender(ControlEvent $event): void
	{
		if (EshopSalesConfig::load('front.allowTagSalesPrice')) {
			/** @var ProductPreview $control */
			$control = $event->control;

			$product = $control->getProduct();

			if ($product) {
				$this->orderSales->loadProductTagSalePrice($product);
			}
		}
	}

	public function beforeRender(ControlEvent $event): void
	{
		if (EshopSalesConfig::load('front.allowTagSalesPrice')) {
			/** @var DefaultPresenter $presenter */
			$presenter = $event->control;

			if ($presenter->getAction(true) === ':EshopCatalog:Front:Default:product' && $presenter->product) {
				$this->orderSales->loadProductTagSalePrice($presenter->product);
			}
		}
	}

	public function orderCartDetailRenderSummary(ComponentTemplateEvent $event): void
	{
		$this->setTemplateNextOrderSales($event);
	}

	public function orderCartDetailRender(ComponentTemplateEvent $event): void
	{
		$this->setTemplateNextOrderSales($event);
	}

	public function cartDetailRender(ComponentTemplateEvent $event): void
	{
		$this->setTemplateNextOrderSales($event);
	}

	public function orderSummaryRender(ComponentTemplateEvent $event): void
	{
		$this->setTemplateNextOrderSales($event);
	}

	protected function setTemplateNextOrderSales(ComponentTemplateEvent $event): void
	{
		if ($this->config->get('cartComponentsLoadNextOrderSales') !== true)
			return;

		$event->template->nextOrderSales = $this->orderSales->getNextSales($this->cartService->getCurrentCart()->getCartItemsPrice());
	}
}
