<?php declare(strict_types = 1);

namespace EshopSales\FrontModule\Model\Subscribers;

use Core\Model\Event\Event;
use Currency\Model\Exchange;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\FrontModule\Components\Order\OrderForm;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use EshopOrders\FrontModule\Model\Dao\Spedition;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Payments;
use EshopOrders\FrontModule\Model\Speditions;
use EshopOrders\Model\Entities\OrderItem;
use EshopSales\Model\Entities\OrderSale;
use EshopSales\Model\EshopSalesConfig;
use Nette\Security\User;
use Nette\Utils\Arrays;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopSales\FrontModule\Model\OrderSales;
use Core\Model\Entities\EntityManagerDecorator;
use Contributte\Translation\Translator;

class OrderFormSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected OrderSales             $orderSales;
	protected Carts                  $cartsService;
	protected Translator             $translator;
	protected Payments               $paymentsService;
	protected Speditions             $speditionsService;
	protected Exchange               $exchange;
	protected User                   $user;
	protected Customers              $customers;
	protected ProductsFacade         $productsFacade;

	public function __construct(
		EntityManagerDecorator $em,
		OrderSales             $orderSales,
		Carts                  $carts,
		Translator             $translator,
		Payments               $paymentsService,
		Speditions             $speditionsService,
		Exchange               $exchange,
		ProductsFacade         $productsFacade,
		User                   $user,
		Customers              $customers
	)
	{
		$this->em                = $em;
		$this->orderSales        = $orderSales;
		$this->cartsService      = $carts;
		$this->translator        = $translator;
		$this->paymentsService   = $paymentsService;
		$this->speditionsService = $speditionsService;
		$this->exchange          = $exchange;
		$this->productsFacade    = $productsFacade;
		$this->user              = $user;
		$this->customers         = $customers;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderBeforeSave'         => ['orderBeforeSave', 100],
			'eshopOrders.saveOrderFormDataStep3'  => ['orderFormSaveStep3', 100],
			'eshopOrders.orderFormValidate'       => ['orderFormValidate', 100],
			OrderForm::class . '::loadSpeditions' => ['loadSpeditions', 100],
		];
	}

	public function loadSpeditions(Event $event): void
	{
		$cart = $this->cartsService->getCurrentCart();

		// Discount spedition
		foreach ($cart->getDiscounts() as $discount) {
			if ($discount->getType() === OrderSale::TYPE_DELIVERY_PRICE) {
				foreach ($event->data['speditions'] as $country => $rows) {
					foreach ($rows as $k => $v) {
						/** @var Spedition $v */
						if ($v->priceInBaseCurrency >= $discount->amountInBaseCurrency) {
							$v->priceInBaseCurrency = $discount->amountInBaseCurrency;
							$v->price               = $discount->amount;
						}
					}
				}

				break;
			}

			if ($discount->getType() === OrderSale::TYPE_DELIVERY_PRICE_FIRST_ORDER) {
				foreach ($event->data['speditions'] as $country => $rows) {
					foreach ($rows as $k => $v) {
						/** @var Spedition $v */
						if ($v->priceInBaseCurrency >= $discount->amountInBaseCurrency) {
							$v->priceInBaseCurrency = $discount->amountInBaseCurrency;
							$v->price               = $discount->amount;
						}
					}
				}

				break;
			}
		}

		$discounts = array_filter($cart->getCartItems(), static fn(CartItem $ci) => $ci->getProduct() !== null && $ci->getProduct()->isDiscount);
		$isEveryCartItemDiscount = Arrays::every($cart->getCartItems(), static fn(CartItem $ci) => $ci->getProduct() !== null && $ci->getProduct()->isDiscount);
		$isEveryDiscountByEmail = Arrays::every($discounts, static fn(CartItem $ci) => $ci->getProduct() !== null && $ci->getProduct()->getMoreDataValue('discountShipmentMethod') !== 'physical');
		foreach ($event->data['speditions'] as $country => $speditions) {
			/** @var Spedition $spedition */
			foreach ($speditions as $k => $spedition) {
				if (!$discounts || !$isEveryCartItemDiscount || !$isEveryDiscountByEmail) { // v kosiku neni zadny slevovy poukaz nebo je alespon jeden poukaz je k zaslani fyzicky nebo je tam mix poukazu a produktu, odebirame moznost dopravy emailem
					if ($spedition->getIdent() === 'email') {
						unset($event->data['speditions'][$country][$k]);
					}
				} else {
					if ($spedition->getIdent() !== 'email') {
						unset($event->data['speditions'][$country][$k]);
					}
				}
			}
		}
	}

	public function orderFormSaveStep3(SaveOrderFormDataEvent $event): void
	{
		$cart = $this->cartsService->getCurrentCart();

		foreach ($cart->getDiscounts() as $k => $v) {
			if ($v->packageType !== 'eshopSales') {
				continue;
			}

			if ($v->type === OrderSale::TYPE_DELIVERY_PRICE_FIRST_ORDER && !$this->orderSales->canUseOnFirstOrder(
					$event->data['email'] ?? null,
					$event->data['phone'] ?? null
				)) {
				$cart->removeDiscount($v->id);
				$this->orderSales->removeFromCartByCode($v->id);

				$event->form->addError(
					$this->translator->translate('eshopSalesFront.errors.canUseOnlyForFirstOrder', [
						'code' => $v->id,
					]),
					false
				);
			}
		}
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		$cart  = $this->cartsService->getCurrentCart();
		$order = $event->order;

		$this->orderSales->setAutoSalesToCart($cart);

		foreach ($cart->getDiscounts() as $k => $v) {
			if ($v->packageType !== 'eshopSales') {
				continue;
			}

			$discount = new OrderDiscount($v->id, $order);
			$discount->setValue((float) $v->amountInBaseCurrency);
			$discount->setType($v->type);
			$discount->setPrice($v->discountInBaseCurrency);
			$discount->setName($v->text ?: $v->id . ' - ' . $this->translator->translate('eshopOrders.discount') . ': ' . $v->amount . $v->typeSymbol);
			$discount->setDescription($v->description);
			$this->em->persist($discount);
			$order->addOrderDiscount($discount);

			if ($v->getType() === OrderSale::TYPE_RANDOM_CAT_PRODUCT) {
				$ids = $v->getValue() ? $this->productsFacade->productsService->getProductsIdInCategory((int) $v->getValue()) : [];

				if ($ids) {
					$id      = $ids[array_rand($ids)];
					$product = $this->productsFacade->getProduct((int) $id);

					if ($product) {
						/** @var Product $productEntity */
						$productEntity = $this->em->getReference(Product::class, $product->getId());

						$orderItem = new OrderItem($productEntity, $order);
						$orderItem->setQuantity(1);
						$orderItem->setPrice(0);
						$orderItem->recyclingFee = $product->recyclingFee;
						$orderItem->setVatRate($product->getVatRate());
						$orderItem->setCode1($product->code1);

						$orderItem->addOrderItemText($this->translator->getLocale());
						$orderItem->getOrderItemText($this->translator->getLocale())
							->setName($this->translator->translate('eshopSalesFront.discountsGrid.freeProduct') . ': ' . $product->getName());

						$this->em->persist($orderItem);
						$order->getOrderItems()->add($orderItem);
					}
				}
			} else if ($v->getType() === OrderSale::TYPE_PRODUCT) {
				$product = $this->productsFacade->getProduct((int) $v->getValue());

				if ($product) {
					/** @var Product $productEntity */
					$productEntity = $this->em->getReference(Product::class, $product->getId());

					$orderItem = new OrderItem($productEntity, $order);
					$orderItem->setQuantity(1);
					$orderItem->setPrice(0);
					$orderItem->recyclingFee = $product->recyclingFee;
					$orderItem->setVatRate($product->getVatRate());
					$orderItem->setCode1($product->code1);

					$orderItem->addOrderItemText($this->translator->getLocale());
					$orderItem->getOrderItemText($this->translator->getLocale())
						->setName($this->translator->translate('eshopSalesFront.discountsGrid.freeProduct') . ': ' . $product->getName());

					$this->em->persist($orderItem);
					$order->getOrderItems()->add($orderItem);
				}
			}
		}
	}

	public function orderFormValidate(OrderFormEvent $event): void
	{
		if (EshopSalesConfig::load('autoGenerateDiscountCoupon.enable')) {
			$cart        = $this->cartsService->getCurrentCart();
			$values      = $event->form->getValues();
			$payments    = (array) $values['payments'];
			$paymentId   = $payments[array_key_first($payments)];
			$speditions  = (array) $values['speditions'];
			$speditionId = $speditions[array_key_first($speditions)];
			if ($paymentId && $speditionId) {
				$cartItems = $cart->getCartItems();
				$payment   = $this->paymentsService->get($paymentId);
				$spedition = $this->speditionsService->get($speditionId);

				$discounts = array_filter($cartItems, static fn(CartItem $ci) => $ci->getProduct() !== null && $ci->getProduct()->isDiscount);
				$isEveryDiscountByEmail = Arrays::every($discounts, static fn(CartItem $ci) => $ci->getProduct() !== null && $ci->getProduct()->getMoreDataValue('discountShipmentMethod') !== 'physical');
				$isEveryDiscount = Arrays::every($cartItems, static fn(CartItem $ci) => $ci->getProduct() !== null && $ci->getProduct()->isDiscount);
				if ($isEveryDiscount && $payment->ident === 'cod' && !EshopSalesConfig::load('autoGenerateDiscountCoupon.allowCodPayment')) {
					$event->form->addError('eshopSalesFront.errors.disallowCodPayment');
				}

				if ($isEveryDiscount && $isEveryDiscountByEmail && $spedition->ident !== 'email' && EshopSalesConfig::load('autoGenerateDiscountCoupon.forceEmailSpedition')) {
					$event->form->addError('eshopSalesFront.errors.allowOnlyEmailSpedition');
				} else if (!$isEveryDiscount && !$isEveryDiscountByEmail && $spedition->ident === 'email' && EshopSalesConfig::load('autoGenerateDiscountCoupon.forceEmailSpedition')) {
					$event->form->addError('eshopSalesFront.errors.disallowEmailSpedition');
				}
			}
		}
	}
}
