<?php declare(strict_types = 1);

namespace EshopSales\Model\Subscribers;

use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Arrays;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderGift;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Statuses;
use EshopOrders\Model\Subscribers\OrderSubscriber;
use EshopSales\AdminModule\Model\OrderSales;
use EshopSales\Model\Entities\OrderSale;
use EshopSales\Model\EshopSalesConfig;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class OrderStatusSubscriber implements EventSubscriberInterface
{
	protected OrderSales      $orderSales;
	protected Statuses        $statusesService;
	protected EventDispatcher $eventDispatcher;

	public function __construct(
		OrderSales      $orderSales,
		Statuses        $statusesService,
		EventDispatcher $eventDispatcher
	)
	{
		$this->orderSales      = $orderSales;
		$this->statusesService = $statusesService;
		$this->eventDispatcher = $eventDispatcher;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			Order::class . '::changeStatus' => ['orderChangeStatus', 101],
		];
	}

	public function orderChangeStatus(OrderStatusEvent $event): void
	{
		if ($event->status !== OrderStatus::STATUS_IS_PAID) {
			return;
		}

		if (EshopSalesConfig::load('autoGenerateDiscountCoupon.enable', false)) {

			// ---------------- vygenerovani slev. kuponu
			$orderSales = [];
			$orderItems = array_filter($event->order->getOrderItems()->toArray(), static fn(OrderItem $oi) => $oi->getProduct() && $oi->getProduct()->isDiscount);
			/** @var OrderItem $orderItem */
			foreach ($orderItems as $orderItem) {
				$product           = $orderItem->getProduct();
				$expiration        = (int) EshopSalesConfig::load('autoGenerateDiscountCoupon.expiration');
				$dateFrom          = (new DateTime)->setTime(0, 0);
				$dateTo            = ((new DateTime)->modify("+{$expiration} days"))->setTime(0, 0);
				$amount            = $product->discountValue ?: $product->price;
				$isActive          = true;
				$repetitions       = (int) EshopSalesConfig::load('autoGenerateDiscountCoupon.repetitions', 1);
				$bulkCreationCount = $orderItem->getQuantity();
				$type              = $product->discountType ?: OrderSale::TYPE_FIX;

				$orderSales = array_merge($this->orderSales->save([
					'type'                          => $type,
					'dateFrom'                      => $dateFrom,
					'dateTo'                        => $dateTo,
					'amount'                        => (float) $amount,
					'fromPrice'                     => (float) $amount, // od ceny bude stejné jako od hodnoty kuponu
					'isActive'                      => $isActive,
					'code'                          => null,
					'maxRepetitions'                => $repetitions,
					'bulkCreationCount'             => $bulkCreationCount,
					'orderItem'                     => $orderItem,
					'orderGift'                     => null,
					'editedId'                      => null,
					'autoGenerateCodeIfCodeIsEmpty' => true,
					'description'                   => null,
				]), $orderSales);
			}

			/** @var OrderSale[] $orderSaleGifts */
			$orderSaleGifts = [];
			$orderGifts = array_filter($event->order->getGifts()->toArray(), static fn(OrderGift $og) => $og->getProduct() && $og->getProduct()->isDiscount);
			/** @var OrderGift $orderGift */
			foreach ($orderGifts as $orderGift) {
				$product           = $orderGift->getProduct();
				$expiration        = (int) EshopSalesConfig::load('autoGenerateDiscountCoupon.asGifts.expiration');
				$dateFrom          = (new DateTime)->setTime(0, 0);
				$dateTo            = ((new DateTime)->modify("+{$expiration} days"))->setTime(0, 0);
				$amount            = $product->discountValue ?: $product->price;
				$isActive          = true;
				$repetitions       = (int) EshopSalesConfig::load('autoGenerateDiscountCoupon.repetitions', 1);
				$bulkCreationCount = 1;
				$type              = $product->discountType ?: OrderSale::TYPE_FIX;

				$orderSaleGifts = array_merge($this->orderSales->save([
					'type'                          => $type,
					'dateFrom'                      => $dateFrom,
					'dateTo'                        => $dateTo,
					'amount'                        => (float) $amount,
					'fromPrice'                     => $type === OrderSale::TYPE_FIX ? ((float) $amount) : 0.1,
					'isActive'                      => $isActive,
					'code'                          => null,
					'maxRepetitions'                => $repetitions,
					'bulkCreationCount'             => $bulkCreationCount,
					'orderItem'                     => null,
					'orderGift'                     => $orderGift,
					'editedId'                      => null,
					'autoGenerateCodeIfCodeIsEmpty' => true,
					'description'                   => null,
				]), $orderSaleGifts);
			}

			$afterSendSetExpedition = EshopSalesConfig::load('autoGenerateDiscountCoupon.afterSendSetExpedition', false);
			$afterSendSetFinished   = EshopSalesConfig::load('autoGenerateDiscountCoupon.afterSendSetFinished', false);
			$orderStatusSpedition   = null;
			$orderStatusFinished    = null;
			if ($orderSales && $event->order->isPaid && ($afterSendSetExpedition || $afterSendSetFinished)) {
				$isEveryOrderItemDiscount = Arrays::every($event->order->getOrderItems()->toArray(), static fn(OrderItem $oi) => $oi->getProduct() !== null && $oi->getProduct()->isDiscount);
				$isEveryDiscountByEmail   = Arrays::every($orderItems, static fn(OrderItem $oi) => $oi->getProduct() !== null && $oi->getProduct()->getMoreDataValue('discountShipmentMethod') !== 'physical');

				if ($isEveryOrderItemDiscount && $isEveryDiscountByEmail) {
					if ($afterSendSetExpedition && !$event->order->isShipped()) {
						$statusSpedition = $this->statusesService->get(OrderStatus::STATUS_SPEDITION);
						$orderStatusSpedition = new OrderStatus($event->order, $statusSpedition);
						$event->order->getOrderStatuses()->add($orderStatusSpedition);
					}

					if ($afterSendSetFinished && !$event->order->isFinished()) {
						$statusFinished = $this->statusesService->get(OrderStatus::STATUS_FINISHED);
						$orderStatusFinished = new OrderStatus($event->order, $statusFinished);
						$event->order->getOrderStatuses()->add($orderStatusFinished);
					}
				}
			}

			// ---------------- odeslani slev. kuponu na emaily
			$orderSalesToSend = array_filter($orderSales, static function(OrderSale $orderSale) {
				$sendPhysicalToEmail = EshopSalesConfig::load('autoGenerateDiscountCoupon.sendPhysicalToEmail', false);

				if ($sendPhysicalToEmail) {
					return true;
				}

				$orderItem = $orderSale->orderItem;
				/** @phpstan-ignore-next-line */
				if ($orderItem && $orderItem->getProduct() && $orderItem->getProduct()->getMoreDataValue('discountShipmentMethod') === 'physical') {
					return false;
				}

				return true;
			});

			foreach ($orderSaleGifts as $os) {
				if (EshopSalesConfig::load('autoGenerateDiscountCoupon.sendPhysicalToEmail', false)) {
					$orderSalesToSend[] = $os;
					continue;
				}

				$orderGift = $os->orderGift;
				/** @phpstan-ignore-next-line */
				if ($orderGift && $orderGift->getProduct() && $orderGift->getProduct()->getMoreDataValue('discountShipmentMethod') === 'physical') {
					$orderSalesToSend[] = $os;
				}
			}

			if ($orderSalesToSend) {
				$this->orderSales->sendDiscountCoupons($orderSalesToSend);
			}

			if ($orderStatusSpedition) {
				OrderSubscriber::$enableSendInvoice = true;
				$this->statusesService->sendOrderStatusEmail($event->order, $orderStatusSpedition);
			}

			if ($orderStatusFinished) {
				OrderSubscriber::$enableSendInvoice = true;
				$this->statusesService->sendOrderStatusEmail($event->order, $orderStatusFinished);
			}
		}
	}

}