<?php declare(strict_types = 1);

namespace EshopSales\AdminModule\Components\Order;

use Core\AdminModule\Model\Sites;
use Core\Model\Entities\QueryBuilder;
use Core\Model\Templating\Filters\Price;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Doctrine\DBAL\ArrayParameterType;
use EshopOrders\Model\Entities\IDiscount;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Utils\Helpers;
use EshopSales\AdminModule\Model\OrderSales;
use EshopSales\Model\Entities\OrderSale;
use EshopSales\Model\Entities\UsedOrderSale;
use EshopSales\Model\EshopSalesConfig;
use Nette\Utils\Html;

class OrderSalesGrid extends BaseControl
{
	public function __construct(
		protected OrderSales $orderSales,
		protected Helpers    $helpers,
		protected Price      $priceFilter,
		protected Sites      $adminSites,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */

	/**
	 * @param int|string $id
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->presenter;

		if ($this->orderSales->remove($id)) {
			$presenter->flashMessageSuccess('default.deleted');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->orderSales->getEr()->createQueryBuilder('os');

		$dataSource = new DoctrineDataSource($qb, 'os.id');

		$usedCodes = [];
		$inSites   = [];

		$dataSource->onDataLoaded[] = function(array $data) use (&$usedCodes, &$inSites): void {
			/** @var OrderSale[] $data */
			$conn  = $this->em->getConnection();
			$ids   = array_map(static fn($v): ?int => $v->getId(), $data);
			$codes = array_map(static fn($v): ?string => $v->code, $data);

			if ($codes !== []) {
				foreach ($conn->executeQuery("SELECT DISTINCT code FROM eshop_sales__used_order_sale WHERE code IN ('" . implode("','", $codes) . "')", ['codes' => $codes], ['codes' => ArrayParameterType::STRING])
					         ->iterateAssociative() as $v) {
					$usedCodes[$v['code']] = $v['code'];
				}
			}

			if ($ids !== []) {
				foreach ($conn->executeQuery("SELECT DISTINCT order_sale_id, site_id FROM eshop_sales__order_sale_in_site WHERE order_sale_id IN (:ids)", ['ids' => $ids], ['ids' => ArrayParameterType::INTEGER])
					         ->iterateAssociative() as $v) {
					$inSites[$v['order_sale_id']][] = $v['site_id'];
				}
			}
		};

		$grid->setDataSource($dataSource);

		$grid->setDefaultPerPage(20);
		$grid->defaultSort = ['dateFrom' => 'asc'];

		// Columns
		$grid->addColumnText('code', 'eshopSales.orderSale.code')
			->setFilterText();
		$grid->addColumnText('type', 'eshopSales.orderSale.type')
			->setRenderer(fn(OrderSale $row): string => $this->t(OrderSale::getTypesOptions()[$row->getType()]))
			->setFilterSelect(['' => ''] + OrderSale::getTypesOptions())
			->setTranslateOptions();
		$grid->addColumnText('amount', 'eshopSales.orderSale.amount')
			->setRenderer(fn(IDiscount $row): string => $this->helpers->formatDiscount($row))
			->setAlign('right');
		$grid->addColumnPrice('fromPrice', 'eshopSales.orderSale.fromPrice');
		$grid->addColumnDateTime('dateFrom', 'eshopSales.orderSale.dateFrom')
			->setSortable();
		$grid->addColumnDateTime('dateTo', 'eshopSales.orderSale.dateTo')
			->setSortable();

		if (count($this->adminSites->getAll()) > 1) {
			$grid->addColumnText('sites', 'eshopSales.orderSale.sites')
				->setRenderer(function(OrderSale $row) use (&$inSites) {
					return implode(', ', $inSites[$row->getId()] ?? []);
				});
		}

		if (EshopSalesConfig::load('orderSalesGrid.showDescription')) {
			$descriptionColumn = $grid->addColumnText('description', 'eshopSales.orderSale.description')
				->setSortable();
			$descriptionColumn->setFilterText();
		}

		$usedColumn = $grid->addColumnText('used', 'eshopSales.orderSale.used')
			->setRenderer(function(OrderSale $row) use (&$usedCodes) {
				$wrap = Html::el('span', [
					'class' => 'btn btn-xs',
				]);

				if (isset($usedCodes[$row->code])) {
					$wrap->addClass('btn-success')
						->addHtml(Html::el('i', ['class' => 'fas fa-check']));
				} else {
					$wrap->addClass('btn-danger')
						->addHtml(Html::el('i', ['class' => 'fas fa-times']));
				}

				return $wrap;
			});
		$usedColumn->setFitContent(true)
			->setAlign('center');
		$usedColumn->setFilterSelect([
			'' => '',
			1  => 'default.yes',
			0  => 'default.no',
		])->setTranslateOptions()->setCondition(function(QueryBuilder $qb, $value): void {
			$subQb = $this->em->createQueryBuilder()->select('DISTINCT uos.code')
				->from(UsedOrderSale::class, 'uos');

			if ((int) $value === 1) {
				$qb->andWhere($qb->expr()->in('os.code', $subQb->getDQL()));
			} else {
				$qb->andWhere($qb->expr()->notIn('os.code', $subQb->getDQL()));
			}
		});

		$isActiveColumn = $grid->addColumnStatus('isActive', 'default.isActive');
		$isActiveColumn->setAlign('center');
		$isActiveColumn->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = $this->gridPublishChange(...);
		$isActiveColumn->setFilterSelect(['' => ''] + [1 => 'default.yes', 0 => 'default.no'])
			->setTranslateOptions();

		if (self::allowSaleDiscountCouponsAsProducts()) {
			$grid->addColumnText('puchasedIn', 'eshopSales.orderSale.puchasedIn')
				->setFitContent()
				->setAlign('center')
				->setTemplateEscaping(false)
				->setRenderer(function(OrderSale $orderSale) {
					$html = Html::el();
					if ($orderSale->orderItem instanceof OrderItem) {
						$oId = $orderSale->orderItem->order->getId();
						$html->addHtml(
							Html::el('a', [
								'href'   => $this->presenter->link(':EshopOrders:Admin:Default:editOrder', $oId),
								'target' => '_blank',
							])->setText((int) $oId)
						);
					}

					return $html;
				});
		}

		// Prototype
		$grid->getColumn('isActive')->getElementPrototype('td')->addClass('w1nw');

		// Actions
		if (self::allowSaleDiscountCouponsAsProducts()) {
			$grid->addAction('preview', '', ':discountCouponPdf')
				->setRenderCondition(fn(OrderSale $os): bool => $os->orderItem instanceof OrderItem)
				->setIcon('eye')
				->setOpenInNewTab()
				->setBsType();
		}

		$grid->addAction('edit', '', ':editOrderSale!')->setIcon('pencil-alt')->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	/**
	 * @param int|string $id
	 */
	public function gridPublishChange($id, string $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->orderSales->setActive((int) $id, (int) $newStatus)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	protected static function allowSaleDiscountCouponsAsProducts(): bool
	{
		return (bool) EshopSalesConfig::loadScalar('autoGenerateDiscountCoupon.enable');
	}

}
