<?php declare(strict_types = 1);

namespace EshopSales\FrontModule\Components\Order;

use Core\Model\Event\Event;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\TextInput;
use EshopOrders\FrontModule\Model\CartHelper;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Customers;
use EshopSales\FrontModule\Model\OrderSales;
use EshopSales\Model\Config;
use EshopSales\Model\Entities\OrderSale;
use EshopSales\Model\EshopSalesConfig;
use Exception;
use Nette\Http\SessionSection;
use Nette\Security\IIdentity;
use Nette\Utils\ArrayHash;
use Users\Model\Entities\User;


class DiscountForm extends BaseControl
{
	protected SessionSection $sessionSection;

	public function __construct(
		protected OrderSales                 $orderSales,
		protected Config                     $config,
		private readonly Carts               $cartsService,
		protected CartHelper                 $cartHelperService,
		protected \Users\Model\Security\User $user,
		protected Customers                  $customersService
	)
	{
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax()
			->setShowLangSwitcher(false);

		$form->addText('discountCode', 'eshopOrders.orderPage.discountCode')
			->setRequired();

		$form->addSubmit('submit', 'eshopSalesFront.discountForm.submit', 'darkest-grey-bg-color dark-grey-bg-hover-color');

		$form->onValidate[] = $this->formOnValidate(...);
		$form->onSuccess[]  = $this->formOnSuccess(...);

		return $form;
	}

	/**
	 *
	 * @throws Exception
	 */
	public function formOnValidate(BaseForm $form, ArrayHash $values): void
	{
		$cart = $this->cartsService->getCurrentCart();

		/** @var User $user */
		$user     = $this->user->getIdentity();
		$customer = $this->user->getIdentity() instanceof IIdentity ? $this->customersService->getByUser($user) : null;

		/** @var TextInput $discountCodeControl */
		$discountCodeControl = $form['discountCode'];

		$cartItemsPrice          = $cart->getCartItemsPrice();
		$cartSales               = $this->orderSales->getOrderSalesFromCart($cartItemsPrice);
		$cartItemsPriceAfterSale = $cartItemsPrice - abs($cart->calculateDiscounts());

		$formSale = $this->orderSales->isValidDiscountCode($values->discountCode, $cartItemsPriceAfterSale);

		if ($formSale instanceof \EshopSales\FrontModule\Model\Dao\OrderSale) {
			$productsCanUse = false;
			foreach ($cart->getCartItems() as $item) {
				if (
					!$item->product
					|| $formSale->canUseOnProduct($item->product, $customer)
				) {
					$productsCanUse = true;
					break;
				}
			}

			if (!$productsCanUse) {
				$discountCodeControl->addError('eshopSalesFront.discountForm.productsForCodeNotFound');
			}

			if (!EshopSalesConfig::load('front.allowForGroupCustomers', true) && ($customer && $customer->getGroupCustomers()
					&& ($formSale->customerGroups === [] && !in_array($customer->getGroupCustomers()->getId(), $formSale->customerGroups)))) {
				$discountCodeControl->addError('eshopSalesFront.discountForm.notAllowedForGroupCustomer');
			}

			if ($formSale->type === OrderSale::TYPE_PERCENT) {
				foreach ($cartSales as $sale) {
					if ($sale->type === OrderSale::TYPE_PERCENT) {
						$discountCodeControl->addError('eshopSalesFront.discountForm.onlyOnePercentCode');
						break;
					}
				}
			} else if ($formSale->type === OrderSale::TYPE_DELIVERY_PRICE_FIRST_ORDER) {
				if (!$this->orderSales->canUseOnFirstOrder()) {
					$discountCodeControl->addError('eshopSalesFront.discountForm.canUseOnlyForFirstOrder');
				}
			}

			if (count($cartSales) === $this->config->get('discountCouponsCanBeRedeemedAtOnce')) {
				$discountCodeControl->addError('eshopSalesFront.discountForm.discountCodesExhausted');
			}
		} else {
			$discountCodeControl->addError('eshopSalesFront.discountForm.discountCodeInvalid');
		}

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formOnSuccess(BaseForm $form, ArrayHash $values): void
	{
		$orderSale = $this->orderSales->getByCode($values->discountCode);

		if (!$orderSale instanceof \EshopSales\FrontModule\Model\Dao\OrderSale) {
			return;
		}

		$this->orderSales->addOrderSaleToCart($orderSale->id);
		$this->eventDispatcher->dispatch(new Event(), 'eshopSales.discountCodesChanged');
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}
}

