<?php declare(strict_types = 1);

namespace EshopSales\Model\Entities;

use Core\Model\Entities\Site;
use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Entities\IDiscount;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderGift;
use EshopOrders\Model\Entities\OrderItem;
use EshopSales\Model\EshopSalesConfig;
use EshopSales\Model\Listeners\OrderSaleListener;
use InvalidArgumentException;
use Nette\Utils\Validators;

#[ORM\Table('eshop_sales__order_sale')]
#[ORM\Entity]
#[ORM\EntityListeners([OrderSaleListener::class])]
class OrderSale implements IDiscount
{
	use TId;

	/** @var string */
	public const AUTO_SALE_ID = 'ESHOPSALESAUTO';

	const TYPE_FIX                        = 'fix';
	const TYPE_PERCENT                    = 'percent';
	const TYPE_DELIVERY_PRICE             = 'deliveryPrice';
	const TYPE_DELIVERY_PRICE_FIRST_ORDER = 'deliveryPriceFirstOrder';
	const TYPE_RANDOM_CAT_PRODUCT         = 'randomCategoryProduct';
	const TYPE_PRODUCT                    = 'product';

	const TYPES = [
		self::TYPE_FIX                        => [
			'title'  => 'eshopSales.types.' . self::TYPE_FIX,
			'symbol' => '',
		],
		self::TYPE_PERCENT                    => [
			'title'  => 'eshopSales.types.' . self::TYPE_PERCENT,
			'symbol' => '%',
		],
		self::TYPE_DELIVERY_PRICE             => [
			'title'  => 'eshopSales.types.' . self::TYPE_DELIVERY_PRICE,
			'symbol' => '',
		],
		self::TYPE_DELIVERY_PRICE_FIRST_ORDER => [
			'title'  => 'eshopSales.types.' . self::TYPE_DELIVERY_PRICE_FIRST_ORDER,
			'symbol' => '',
		],
		self::TYPE_RANDOM_CAT_PRODUCT         => [
			'title'  => 'eshopSales.types.' . self::TYPE_RANDOM_CAT_PRODUCT,
			'symbol' => '',
		],
		self::TYPE_PRODUCT                    => [
			'title'  => 'eshopSales.types.' . self::TYPE_PRODUCT,
			'symbol' => '',
		],
	];

	#[ORM\Column(name: 'code', type: Types::STRING, nullable: true)]
	public ?string $code = null;

	#[ORM\Column(name: 'is_active', type: Types::SMALLINT, nullable: false, options: ['default' => 1])]
	public int $isActive = 1;

	#[ORM\Column(name: 'type', type: Types::STRING, nullable: false, options: ['default' => 'price'])]
	protected string $type;

	/**
	 * @var float|string
	 */
	#[ORM\Column(name: 'amount', type: Types::DECIMAL, precision: 10, scale: 3, nullable: false)]
	protected $amount;

	/**
	 * @var float|string
	 */
	#[ORM\Column(name: 'from_price', type: Types::DECIMAL, precision: 10, scale: 3, nullable: false)]
	protected $fromPrice;

	#[ORM\Column(name: 'date_from', type: Types::DATETIME_MUTABLE, nullable: true)]
	protected ?DateTime $dateFrom = null;

	#[ORM\Column(name: 'date_to', type: Types::DATETIME_MUTABLE, nullable: true)]
	protected ?DateTime $dateTo = null;

	#[ORM\Column(type: Types::INTEGER, nullable: true)]
	public ?int $maxRepetitions = null;

	#[ORM\Column(type: Types::INTEGER, nullable: true)]
	public ?int $currentRepetitions = null;

	#[ORM\ManyToOne(targetEntity: OrderItem::class)]
	#[ORM\JoinColumn(name: 'order_item_id', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	public ?OrderItem $orderItem = null;

	#[ORM\ManyToOne(targetEntity: OrderGift::class)]
	#[ORM\JoinColumn(name: 'order_gift_id', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	public ?OrderGift $orderGift = null;

	#[ORM\Column(name: 'description', type: Types::TEXT, nullable: true)]
	public ?string $description;

	/** @var Collection<OrderSaleInSite> */
	#[ORM\OneToMany(targetEntity: OrderSaleInSite::class, mappedBy: 'orderSale', cascade: ['all'], orphanRemoval: true, indexBy: 'site')]
	public Collection $sites;

	#[ORM\Column(name: 'customer_groups', type: Types::STRING, nullable: true)]
	protected ?string $customerGroups = null;

	#[ORM\Column(name: 'manufacturers', type: Types::STRING, nullable: true)]
	protected ?string $manufacturers = null;

	#[ORM\Column(name: 'categories', type: Types::STRING, nullable: true)]
	protected ?string $categories = null;

	#[ORM\Column(name: 'features', type: Types::STRING, nullable: true)]
	protected ?string $features = null;

	#[ORM\Column(name: 'products', type: Types::STRING, nullable: true)]
	protected ?string $products = null;

	public function __construct(string $type, float $amount, float $fromPrice, ?int $maxRepetitions = null)
	{
		$this->setType($type);
		$this->setAmount($amount);
		$this->setFromPrice($fromPrice);

		$this->isActive           = 1;
		$this->maxRepetitions     = $maxRepetitions;
		$this->currentRepetitions = $maxRepetitions;
		$this->sites              = new ArrayCollection;
	}

	public function addSite(Site $site): void
	{
		$this->sites->add(new OrderSaleInSite($this, $site));
	}

	public function setType(string $type): self
	{
		if (!in_array($type, array_keys(self::TYPES)))
			throw new InvalidArgumentException();

		$this->type = $type;

		return $this;
	}

	public function getType(): string { return $this->type; }

	public function getTypeSymbol(): string { return self::TYPES[$this->type]['symbol']; }

	public function setFromPrice(float $fromPrice): self
	{
		$this->fromPrice = Strings::formatEntityDecimal($fromPrice) ?: 0.0;

		return $this;
	}

	public function getFromPrice(): float { return (float) $this->fromPrice; }

	public function setAmount(float $amount): self
	{
		$this->amount = Strings::formatEntityDecimal($amount) ?: 0.0;

		return $this;
	}

	public function getAmount(): float { return (float) $this->amount; }

	public function setDateFrom(?DateTime $from = null): self
	{
		$this->dateFrom = $from;

		return $this;
	}

	public function getDateFrom(): ?DateTime { return $this->dateFrom; }

	public function setDateTo(?DateTime $to = null): self
	{
		$this->dateTo = $to;

		return $this;
	}

	public function getDateTo(): ?DateTime { return $this->dateTo; }

	public static function getTypesOptions(): array
	{
		$result = [
			self::TYPE_FIX     => self::TYPES[self::TYPE_FIX]['title'],
			self::TYPE_PERCENT => self::TYPES[self::TYPE_PERCENT]['title'],
		];

		if (EshopSalesConfig::load('allowedTypes.deliveryPrice', false)) {
			$result[self::TYPE_DELIVERY_PRICE] = self::TYPES[self::TYPE_DELIVERY_PRICE]['title'];
		}

		if (EshopSalesConfig::load('allowedTypes.deliveryPriceFirstOrder', false)) {
			$result[self::TYPE_DELIVERY_PRICE_FIRST_ORDER] = self::TYPES[self::TYPE_DELIVERY_PRICE_FIRST_ORDER]['title'];
		}

		if (EshopSalesConfig::load('allowedTypes.randomCategoryProduct', false)) {
			$result[self::TYPE_RANDOM_CAT_PRODUCT] = self::TYPES[self::TYPE_RANDOM_CAT_PRODUCT]['title'];
		}

		if (EshopSalesConfig::load('allowedTypes.product', false)) {
			$result[self::TYPE_PRODUCT] = self::TYPES[self::TYPE_PRODUCT]['title'];
		}

		return $result;
	}

	public function isUsageRepeatable(): bool
	{
		return $this->isUsageInfinitelyRepeatable() || $this->maxRepetitions > 1;
	}

	public function isUsageInfinitelyRepeatable(): bool
	{
		return $this->maxRepetitions === null;
	}

	public function isCurrentRepeatable(): bool
	{
		return $this->isUsageInfinitelyRepeatable() || $this->maxRepetitions > $this->currentRepetitions;
	}

	public function decreaseCurrentRepetitions(): void
	{
		if ($this->isUsageInfinitelyRepeatable()) {
			return;
		}

		if ($this->maxRepetitions !== null && $this->currentRepetitions === null) {
			$this->currentRepetitions = $this->maxRepetitions;
		}

		$this->currentRepetitions--;
		$this->currentRepetitions = max($this->currentRepetitions, 0);
	}

	public function isAutoSale(): bool
	{
		return Validators::isNone($this->code);
	}

	public function getValue(): float
	{
		return (float) $this->amount;
	}

	public function toUsedOrderSale(Order $appliedInOrder, OrderItem $createdByOrderItem = null): UsedOrderSale
	{
		$uos = new UsedOrderSale(
			$this->type, $this->code, $this->getValue(), $this->getFromPrice(), $appliedInOrder,
			$this->dateFrom, $this->dateTo, $this->description
		);

		if ($createdByOrderItem) {
			$uos->createdByOrderItem = $createdByOrderItem;
		}

		/** @var OrderSaleInSite $item */
		foreach ($this->sites->toArray() as $item) {
			$uos->sites->add(new UsedOrderSaleInSite($uos, $item->site));
		}

		return $uos;
	}

	public function setCustomerGroups(?array $groups): void
	{
		$this->customerGroups = $groups ? implode(',', $groups) : null;
	}

	public function getCustomerGroups(): array { return explode(',', (string) $this->customerGroups); }

	public function setManufacturers(?array $manu): void
	{
		$this->manufacturers = $manu ? implode(',', $manu) : null;
	}

	public function getManufacturers(): array { return explode(',', (string) $this->manufacturers); }

	public function setCategories(?array $cats): void
	{
		$this->categories = $cats ? implode(',', $cats) : null;
	}

	public function getCategories(): array { return explode(',', (string) $this->categories); }

	public function setFeatures(?array $fea): void
	{
		$this->features = $fea ? implode(',', $fea) : null;
	}

	public function getFeatures(): array { return explode(',', (string) $this->features); }

	public function setProducts(?array $prods): void
	{
		$this->products = $prods ? implode(',', array_unique($prods)) : null;
	}

	public function getProducts(): array { return explode(',', (string) $this->products); }
}
