<?php declare(strict_types = 1);

namespace EshopSales\AdminModule\Components\Order;

use Contributte\Scheduler\Helpers\Debugger;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopSales\AdminModule\Model\Mappers\OrderSale\ICsvMapper;
use EshopSales\AdminModule\Model\OrderSales;
use Exception;
use Nette\Forms\Controls\UploadControl;
use Nette\Http\FileUpload;
use Nette\Schema\ValidationException;
use Nette\Utils\ArrayHash;
use Nette\Utils\Validators;

class ImportOrderSaleForm extends BaseControl
{
	protected OrderSales $orderSales;
	protected ICsvMapper $csvMapper;

	public function __construct(
		OrderSales $orderSales,
		ICsvMapper $csvMapper
	)
	{
		$this->orderSales = $orderSales;
		$this->csvMapper  = $csvMapper;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax()
			->setShowLangSwitcher(false);

		$form->addUpload('file', 'eshopSales.importOrderSaleForm.file')
			->addRule($form::MIME_TYPE, null, ['text/csv', 'application/vnd.ms-excel', 'text/plain'])
			->setRequired();

		$form->addSaveCancelControl();

		$form->onSuccess[]  = [$this, 'formSuccess'];
		$form->onValidate[] = [$this, 'formValidate'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		/** @var FileUpload $file */
		$file   = $values->file;
		$errors = $form->getErrors();

		/** @var UploadControl $upload */
		$upload = $form['file'];
		$upload->cleanErrors();
		foreach ($errors as $err) {
			$form->addError($err);
		}

		if (in_array($file->getContentType(), ['text/csv', 'application/vnd.ms-excel',
				'text/plain']) && !Strings::endsWith($file->name, '.csv')) {
			$form->addError(str_replace('%label', $this->t('eshopSales.importOrderSaleForm.file'), $this->t('default.formMessages.mimeType')));
		}

		$this->redrawControl('form');
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		$presenter = $this->presenter;

		$rowCounter = 1;
		try {
			/** @var FileUpload $file */
			$file = $values->file;
			if ($file->hasFile() && $file->isOk()) {

				$this->em->beginTransaction();

				$data = [];
				$fr   = fopen($file->getTemporaryFile(), 'r');

				if ($fr === false) {
					return;
				}

				$separator = ';';
				while (!feof($fr) && $line = fgetcsv($fr, null, $separator)) {
					$data[] = array_map(static fn($lineV) => Strings::toUtf8($lineV), $line);
				}
				fclose($fr);

				$existCodes     = $this->orderSales->getCodes();
				$normalizedData = $this->csvMapper->normalize($data);

				foreach ($normalizedData as $r) {
					if (Validators::isNone($r['code'])) {
						$rowCounter++;
						continue;
					}

					if (Arrays::contains($existCodes, $r['code'])) {
						$form->addError(sprintf('%s %s', $this->t('eshopSales.importOrderSaleForm.errors.code.isNotUnique'), $this->t('eshopSales.importOrderSaleForm.row', null, ['i' => $rowCounter])), false);
						$this->em->rollback();

						return;
					}
					$this->orderSales->save($r);
					$rowCounter++;
				}

				$this->em->commit();

				$presenter->flashMessageSuccess('default.saved');
			}
		} catch (Exception $e) {
			$this->em->rollback();
			if ($e instanceof ValidationException) {
				foreach ($e->getMessageObjects() as $msg) {
					$form->addError(sprintf('%s %s', $this->t(sprintf('%s.%s.%s', 'eshopSales.importOrderSaleForm.errors', Arrays::first($msg->path), $msg->code)), $this->t('eshopSales.importOrderSaleForm.row', null, ['i' => $rowCounter])), false);
				}
			} else {
				Debugger::log($e);
				$form->addError($e->getMessage());
			}
		}

		$this->redrawControl('form');
	}

}
