<?php declare(strict_types = 1);

namespace EshopSales\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopSales\AdminModule\Model\OrderSales;
use EshopSales\Model\EshopSalesConfig;
use Nette\Utils\Html;

class PurchasedDiscountsGrid extends BaseControl
{
	protected OrderSales $orderSales;

	public function __construct(
		OrderSales $orderSales
	)
	{
		$this->orderSales = $orderSales;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		if (EshopSalesConfig::load('purchasedDiscountsGrid.csvExport')) {
			$grid->addExportCsv('CSV Export', 'slevy.csv', 'windows-1250', ';', false, true);
		}

		$grid->setDataSource($this->orderSales->getPurchasedDiscounts());

		$grid->addColumnText('productName', 'eshopSales.purchasedDiscountsGrid.product')
			 ->setFilterText();
		$grid->addColumnText('order', 'eshopSales.purchasedDiscountsGrid.order')
			->setAlign('center')
			->setTemplateEscaping(false)
			->setRenderer(function(array $purchasedDiscount) {
				$html = Html::el();
				$html->addHtml(
					Html::el('a', [
						'href'   => $this->presenter->link(':EshopOrders:Admin:Default:editOrder', $purchasedDiscount['order']),
						'target' => '_blank',
					])->setText($purchasedDiscount['order'])
				);

				return $html;
			})->setFilterText();
		$grid->addColumnText('codes', 'eshopSales.purchasedDiscountsGrid.codes')
			 ->setRenderer(static fn(array $data) => implode(', ', $data['codes']))
			 ->setFilterText()
			 ->setCondition(function(&$item, $value) {
				 $result = [];
				 foreach ($item as $v) {
					 if (in_array($value, $v['codes'], true)) {
						 $result[] = $v;
					 }
				 }
				 return $result;
			 });
		$grid->addColumnText('quantity', 'eshopSales.purchasedDiscountsGrid.quantity')
			 ->setFitContent()
			 ->setFilterText();
		$grid->addColumnText('invoice', 'eshopSales.purchasedDiscountsGrid.invoice')
			 ->setFitContent()
			 ->setFilterText();
		$grid->addColumnDateTime('purchased', 'eshopSales.purchasedDiscountsGrid.purchased')
			 ->setFitContent()
			 ->setAlign('right')
			 ->setFormat('d.m.Y')
			 ->setFilterDateRange();
		$grid->addColumnText('appliedInOrders', 'eshopSales.purchasedDiscountsGrid.appliedInOrders')
			 ->setTemplateEscaping(false)
			 ->setRenderer(function(array $data) {
				 $result = [];
				 foreach ($data['appliedInOrders'] as $v) {
					 $html = Html::el();
					 $html->addHtml(
						 Html::el('a', [
							 'href'   => $this->presenter->link(':EshopOrders:Admin:Default:editOrder', $v),
							 'target' => '_blank',
						 ])->setText($v)
					 );
					 $result[] = $html;
				 }

				 return implode(', ', $result);
			 })->setFilterText()
			->setCondition(function(&$item, $value) {
				$result = [];
				foreach ($item as $v) {
					if (in_array($value, $v['appliedInOrders'])) {
						$result[] = $v;
					}
				}
				return $result;
			});
		$grid->addColumnText('appliedInOrderInvoices', 'eshopSales.purchasedDiscountsGrid.appliedInOrderInvoices')
			 ->setTemplateEscaping(false)
			 ->setRenderer(fn(array $data) => implode(', ', $data['appliedInOrderInvoices']))->setFilterText()
			 ->setCondition(function(&$item, $value) {
				 $result = [];
				 foreach ($item as $v) {
					 if (in_array($value, $v['appliedInOrderInvoices'])) {
						 $result[] = $v;
					 }
				 }
				 return $result;
			 });
		$grid->addColumnText('dateFrom', 'eshopSales.purchasedDiscountsGrid.dateFrom')
			 ->setFitContent()
			 ->setAlign('right')
			 ->setFilterDateRange();
		$grid->addColumnText('dateTo', 'eshopSales.purchasedDiscountsGrid.dateTo')
			 ->setFitContent()
			 ->setAlign('right')
			 ->setFilterDateRange();
		$grid->addColumnText('status', 'eshopSales.purchasedDiscountsGrid.status')
			 ->setRenderer(fn($v) => $this->t('eshopSales.purchasedDiscountsGrid.' . $v['status']))
			 ->setFilterSelect([
				 null            => null,
				 'applied'       => $this->t('eshopSales.purchasedDiscountsGrid.applied'),
				 'notApplied'    => $this->t('eshopSales.purchasedDiscountsGrid.notApplied'),
				 'codeNotExists' => $this->t('eshopSales.purchasedDiscountsGrid.codeNotExists'),
			 ], 'status')
			 ->setCondition(function(&$item, $value) {
				 $result = [];
				 foreach ($item as $v) {
					 if ($v['status'] === $value) {
						 $result[] = $v;
					 }
				 }
				 return $result;
			 });

		return $grid;
	}

}
