<?php declare(strict_types = 1);

namespace EshopSales\AdminModule\Components\Order;

use Core\AdminModule\Model\Sites;
use Core\Model\Entities\QueryBuilder;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Doctrine\DBAL\Connection;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\Model\Entities\IDiscount;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Utils\Helpers;
use EshopSales\AdminModule\Model\UsedOrderSales;
use EshopSales\Model\Entities\OrderSale;
use EshopSales\Model\Entities\UsedOrderSale;
use EshopSales\Model\EshopSalesConfig;
use Nette\Utils\Html;

class UsedOrderSalesGrid extends BaseControl
{
	protected Sites          $adminSites;
	protected UsedOrderSales $usedOrderSales;
	protected Helpers        $helpers;
	protected Orders         $orders;

	public function __construct(
		UsedOrderSales $usedOrderSales,
		Sites          $adminSites,
		Helpers        $helpers,
		Orders         $orders
	)
	{
		$this->usedOrderSales = $usedOrderSales;
		$this->adminSites     = $adminSites;
		$this->helpers        = $helpers;
		$this->orders         = $orders;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		if (EshopSalesConfig::load('usedOrderSalesGrid.csvExport')) {
			$grid->addExportCsv('CSV Export', 'slevy.csv', 'windows-1250');
		}

		$qb = $this->usedOrderSales->getEr()->createQueryBuilder('os');

		$dataSource = new DoctrineDataSource($qb, 'os.id');

		/** @var Order[] $orders */
		$orders  = [];
		$inSites = [];

		$dataSource->onDataLoaded[] = function(array $data) use (&$orders, &$inSites) {
			/** @var UsedOrderSale[] $data */

			$conn      = $this->em->getConnection();
			$ids       = [];
			$ordersIds = [];

			foreach ($data as $row) {
				$ids[] = $row->getId();

				if ($row->order) {
					$ordersIds[] = $row->order->getId();
				}
			}

			if (!empty($ordersIds)) {
				foreach ($this->orders->getFullOrderQb()
					         ->addSelect('ois')
					         ->leftJoin('items.sales', 'ois')
					         ->andWhere('o.id IN (:id)')
					         ->setParameter('id', $ordersIds)
					         ->getQuery()->getResult() as $row) {
					$orders[$row->getId()] = $row;
				}
			}

			if (!empty($ids)) {
				foreach ($conn->executeQuery("SELECT DISTINCT used_order_sale_id, site_id FROM eshop_sales__used_order_sale_in_site WHERE used_order_sale_id IN (:ids)", ['ids' => $ids], ['ids' => Connection::PARAM_INT_ARRAY])
					         ->iterateAssociative() as $v) {
					$inSites[$v['used_order_sale_id']][] = $v['site_id'];
				}
			}
		};

		$grid->setDataSource($dataSource);
		$grid->setDefaultPerPage(20);
		$grid->defaultSort = ['dateFrom' => 'asc'];

		// Columns
		$grid->addColumnText('code', 'eshopSales.orderSale.code')
			->setFilterText();
		$grid->addColumnText('type', 'eshopSales.orderSale.type')
			->setRenderer(function(UsedOrderSale $row) {
				return $this->t(OrderSale::getTypesOptions()[$row->type]);
			})
			->setFilterSelect(['' => ''] + OrderSale::getTypesOptions())
			->setTranslateOptions();
		$grid->addColumnText('amount', 'eshopSales.orderSale.amount')
			->setRenderer(fn(IDiscount $row) => $this->helpers->formatDiscount($row))
			->setFilterText();
		$grid->getColumn('amount')->setAlign('right');
		$grid->addColumnPrice('orderPrice', 'eshopSales.orderSale.orderPrice')
			->setRenderer(function(UsedOrderSale $row) use (&$orders) {
				if ($row->order && isset($orders[$row->order->getId()])) {
					return $orders[$row->order->getId()]->getPrice();
				}

				return '';
			});
		$grid->addColumnDateTime('dateFrom', 'eshopSales.orderSale.dateFrom')
			->setSortable();
		$grid->addColumnDateTime('dateTo', 'eshopSales.orderSale.dateTo')
			->setSortable();
		$grid->addColumnText('applied', 'eshopSales.orderSale.applied')
			->setRenderer(function(UsedOrderSale $orderSale) {
				if ($orderSale->order) {
					return $orderSale->order->getCreatedTime()->format('d.m.Y');
				}

				return '';
			});
		$grid->addColumnText('order', 'eshopSales.orderSale.puchasedIn')
			->setFitContent()
			->setAlign('center')
			->setTemplateEscaping(false)
			->setRenderer(function(UsedOrderSale $orderSale) {
				$html = Html::el();
				if ($orderSale->order !== null) {
					$oId = $orderSale->order->getId();
					$html->addHtml(
						Html::el('a', [
							'href'   => $this->presenter->link(':EshopOrders:Admin:Default:editOrder', $oId),
							'target' => '_blank',
						])->setText($oId)
					);
				}

				return $html;
			})->setFilterText()
			->setCondition(function(QueryBuilder $qb, $value) {
				$qb->andWhere('os.order = :order')->setParameter('order', $value);
			});

		if (count($this->adminSites->getAll()) > 1) {
			$grid->addColumnText('sites', 'eshopSales.orderSale.sites')
				->setRenderer(function(UsedOrderSale $row) use (&$inSites) {
					return implode(', ', $inSites[$row->getId()] ?? []);
				})
				->setFilterSelect(['' => ''] + $this->adminSites->getOptionsForSelect())
				->setCondition(function(QueryBuilder $qb, $value) {
					$qb->innerJoin('os.sites', 'osSite', Join::WITH, 'osSite.site = :site')
						->setParameter('site', $value);
				});
		}

		if (EshopSalesConfig::load('orderSalesGrid.showDescription')) {
			$grid->addColumnText('description', 'eshopSales.orderSale.description')
				->setSortable()
				->setFilterText();
		}

		$grid->addAction('delete', '', 'remove!')
			->setIcon('times')
			->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		return $grid;
	}

	public function handleRemove(int $id): void
	{
		if ($this->usedOrderSales->remove($id)) {
			$this->presenter->flashMessageSuccess('default.removed');
		} else {
			$this->presenter->flashMessageDanger('default.error');
		}

		$this->presenter->redrawControl('flashes');
		$this['grid']->reload();
	}

}
