<?php declare(strict_types = 1);

namespace EshopSales\AdminModule\Presenters;

use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\SaveCancelControl;
use EshopSales\AdminModule\Components\Order\ExportOrderSaleForm;
use EshopSales\AdminModule\Components\Order\IExportOrderSaleFormFactory;
use EshopSales\AdminModule\Components\Order\IImportOrderSaleFormFactory;
use EshopSales\AdminModule\Components\Order\ImportOrderSaleForm;
use EshopSales\AdminModule\Components\Order\IOrderSaleFormFactory;
use EshopSales\AdminModule\Components\Order\IOrderSalesGridFactory;
use EshopSales\AdminModule\Components\Order\IPurchasedDiscountsGridFactory;
use EshopSales\AdminModule\Components\Order\IUsedOrderSalesGridFactory;
use EshopSales\AdminModule\Components\Order\OrderSaleForm;
use EshopSales\AdminModule\Components\Order\OrderSalesGrid;
use EshopSales\AdminModule\Components\Order\PurchasedDiscountsGrid;
use EshopSales\AdminModule\Components\Order\UsedOrderSalesGrid;
use EshopSales\AdminModule\Model\OrderSales;
use EshopSales\Model\Entities\OrderSale;
use EshopSales\Model\EshopSalesConfig;
use EshopSales\Model\Pdf\IDiscountCouponPdfBuilderFactory;

class DefaultPresenter extends BasePresenter
{
	protected const ICON = 'fas fa-money-check-alt';

	protected IDiscountCouponPdfBuilderFactory $discountCouponFactory;
	protected OrderSales                       $orderSales;

	public string $currentTab = 'default';

	public function __construct(
		IDiscountCouponPdfBuilderFactory $discountCouponFactory,
		OrderSales                       $orderSales
	)
	{
		$this->discountCouponFactory = $discountCouponFactory;
		$this->orderSales            = $orderSales;

		parent::__construct();
	}

	protected function startup(): void
	{
		parent::startup();
		$this->setHeader('eshopSales.title.sales', self::ICON);
	}

	public function beforeRender()
	{
		parent::beforeRender();

		$this->template->currentTab = $this->currentTab;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleOpenImportOrderSaleForm(): void
	{
		$this->template->modalTitle = $this->t('eshopSales.title.importOrderSale');
		$this->template->modal      = 'importOrderSaleForm';
		$this->redrawControl('modal');
	}

	public function handleOpenExportOrderSaleForm(): void
	{
		$this->template->modalTitle = $this->t('eshopSales.title.exportOrderSale');
		$this->template->modal      = 'exportOrderSaleForm';
		$this->redrawControl('modal');
	}

	public function handleAddOrderSale(): void
	{
		$this->template->modalTitle       = $this->t(EshopSalesConfig::load('duplicateAddOrderSaleButton', false) ? 'eshopSales.title.addOrderOrVoucherDiscount' : 'eshopSales.title.addOrderSale');
		$this->template->modal            = 'orderSaleForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	public function handleEditOrderSale(int $id): void
	{
		/** @var OrderSale|null $orderSale */
		$orderSale = $this->em->getRepository(OrderSale::class)->find($id);

		if ($orderSale === null) {
			$this->error();
		}

		$this['orderSaleForm']->id = $id;
		$this['orderSaleForm']->setOrderSale($id);
		$this->template->modalTitle       = $this->t('eshopSales.title.editOrderSale');
		$this->template->modal            = 'orderSaleForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	public function handleSendDiscountCoupons(int $orderId): void
	{
		$this->orderSales->sendDiscountCoupons($this->orderSales->getDiscountsPurchasedInOrder($orderId));
		$this->flashMessageSuccess('eshopSales.orderForm.discountCouponsSend');
		$this->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault(): void
	{
		$nav[] = [
			'title' => 'eshopSales.menu.addOrderSale',
			'link'  => $this->link('addOrderSale!'),
			'ico'   => 'plus',
			'class' => 'ajax',
		];

		if (EshopSalesConfig::load('duplicateAddOrderSaleButton', false)) {
			$nav[] = [
				'title' => 'eshopSales.menu.addVoucherDiscount',
				'link'  => $this->link('addOrderSale!'),
				'ico'   => 'plus',
				'class' => 'ajax !btn-primary',
			];
		}

		if (EshopSalesConfig::load('allowImport', false)) {
			$nav[] = [
				'title' => 'eshopSales.menu.importOrderSale',
				'link'  => $this->link('openImportOrderSaleForm!'),
				'ico'   => 'fas fa-upload',
				'class' => 'ajax',
			];
		}

		if (EshopSalesConfig::load('allowExport', false)) {
			$nav[] = [
				'title' => 'eshopSales.menu.exportOrderSale',
				'link'  => $this->link('openExportOrderSaleForm!'),
				'ico'   => 'fas fa-download',
				'class' => 'ajax',
			];
		}

		if (EshopSalesConfig::load('allowPurchasedDiscounts', false)) {
			$nav[] = [
				'title' => 'eshopSales.menu.purchasedDiscounts',
				'link'  => $this->link('purchasedDiscounts'),
				'ico'   => 'clipboard-list',
			];
		}

		$this['navigation']->setData(['header' => $nav]);
	}

	public function actionUsedOrderSales(): void
	{
		$this->currentTab = 'usedOrderSales';

		$this->setTitle('eshopSales.menu.usedOrderSale');
		$this->setHeader('eshopSales.menu.usedOrderSale');
	}

	/**
	 * @param int $id
	 */
	public function actionDiscountCouponPdf($id): void
	{
		/** @var OrderSale|null $orderSale */
		$orderSale = $this->em->getRepository(OrderSale::class)->find($id);

		if ($orderSale === null || $orderSale->orderItem === null) {
			$this->error();
		}

		$this->discountCouponFactory->create($orderSale)->render();
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentUsedOrderSalesGrid(IUsedOrderSalesGridFactory $factory): UsedOrderSalesGrid
	{
		return $factory->create();
	}

	protected function createComponentOrderSaleForm(IOrderSaleFormFactory $factory): OrderSaleForm
	{
		$control = $factory->create();

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[]         = function(BaseForm $form) {
				if ($form->getCustomData('orderSaleId') !== null) {
					$this->handleEditOrderSale((int) $form->getCustomData('orderSaleId'));
				}

				$this->redrawControl('flashes');
				$this['orderSalesGrid']['grid']->reload();
			};
			$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
				$this->payload->hideModal = true;
				$this->redrawControl('flashes');
				$this['orderSalesGrid']['grid']->reload();
			};
			/** @var SaveCancelControl $saveControl */
			$saveControl = $control['form']['saveControl'];
			$saveControl->closeModalOnCancel();
		};

		return $control;
	}

	protected function createComponentImportOrderSaleForm(IImportOrderSaleFormFactory $factory): ImportOrderSaleForm
	{
		$control = $factory->create();

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[]         = function(BaseForm $form) {
				$this->redrawControl('flashes');
				$this['orderSalesGrid']['grid']->reload();
			};
			$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
				$this->payload->hideModal = true;
				$this->redrawControl('flashes');
				$this['orderSalesGrid']['grid']->reload();
			};

			/** @var SaveCancelControl $saveControl */
			$saveControl = $control['form']['saveControl'];
			$saveControl->closeModalOnCancel();
		};

		return $control;
	}

	protected function createComponentExportOrderSaleForm(IExportOrderSaleFormFactory $factory): ExportOrderSaleForm
	{
		$control = $factory->create();

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[]         = function(BaseForm $form) {
				$this->redrawControl('flashes');
			};
			$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
				$this->payload->hideModal = true;
				$this->redrawControl('flashes');
			};

			/** @var SaveCancelControl $saveControl */
			$saveControl = $control['form']['saveControl'];
			$saveControl->closeModalOnCancel();
		};

		return $control;
	}

	protected function createComponentOrderSalesGrid(IOrderSalesGridFactory $factory): OrderSalesGrid
	{
		return $factory->create();
	}

	protected function createComponentPurchasedDiscountsGrid(IPurchasedDiscountsGridFactory $factory): PurchasedDiscountsGrid
	{
		return $factory->create();
	}

}
