<?php declare(strict_types = 1);

namespace EshopSales\Model\Entities;

use Core\Model\Entities\Site;
use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Entities\IDiscount;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopSales\Model\EshopSalesConfig;
use Nette\Utils\Validators;

/**
 * @ORM\Table("eshop_sales__order_sale")
 * @ORM\Entity
 * @ORM\EntityListeners({"EshopSales\Model\Listeners\OrderSaleListener"})
 */
class OrderSale implements IDiscount
{
	use TId;

	/** @var string */
	public const AUTO_SALE_ID = 'ESHOPSALESAUTO';

	const TYPE_FIX                        = 'fix';
	const TYPE_PERCENT                    = 'percent';
	const TYPE_DELIVERY_PRICE             = 'deliveryPrice';
	const TYPE_DELIVERY_PRICE_FIRST_ORDER = 'deliveryPriceFirstOrder';
	const TYPE_RANDOM_CAT_PRODUCT         = 'randomCategoryProduct';
	const TYPE_PRODUCT                    = 'product';

	const TYPES = [
		self::TYPE_FIX                        => [
			'title'  => 'eshopSales.types.' . self::TYPE_FIX,
			'symbol' => '',
		],
		self::TYPE_PERCENT                    => [
			'title'  => 'eshopSales.types.' . self::TYPE_PERCENT,
			'symbol' => '%',
		],
		self::TYPE_DELIVERY_PRICE             => [
			'title'  => 'eshopSales.types.' . self::TYPE_DELIVERY_PRICE,
			'symbol' => '',
		],
		self::TYPE_DELIVERY_PRICE_FIRST_ORDER => [
			'title'  => 'eshopSales.types.' . self::TYPE_DELIVERY_PRICE_FIRST_ORDER,
			'symbol' => '',
		],
		self::TYPE_RANDOM_CAT_PRODUCT         => [
			'title'  => 'eshopSales.types.' . self::TYPE_RANDOM_CAT_PRODUCT,
			'symbol' => '',
		],
		self::TYPE_PRODUCT                    => [
			'title'  => 'eshopSales.types.' . self::TYPE_PRODUCT,
			'symbol' => '',
		],
	];

	/**
	 * @var string|null
	 * @ORM\Column(name="code", type="string", nullable=true)
	 */
	public $code;

	/**
	 * @var bool
	 * @ORM\Column(name="is_active", type="boolean", nullable=false, options={"default": true})
	 */
	public $isActive;

	/**
	 * @var string
	 * @ORM\Column(name="type", type="string", nullable=false, options={"default":"price"})
	 */
	protected $type;

	/**
	 * @var float|string
	 * @ORM\Column(name="amount", type="decimal", precision=10, scale=3, nullable=false)
	 */
	protected $amount;

	/**
	 * @var float|string
	 * @ORM\Column(name="from_price", type="decimal", precision=10, scale=3, nullable=false)
	 */
	protected $fromPrice;

	/**
	 * @var DateTime
	 * @ORM\Column(name="date_from", type="datetime", nullable=true)
	 */
	protected $dateFrom;

	/**
	 * @var DateTime
	 * @ORM\Column(name="date_to", type="datetime", nullable=true)
	 */
	protected $dateTo;

	/**
	 * @var int|null
	 * @ORM\Column(type="integer", nullable=true)
	 */
	public $maxRepetitions;

	/**
	 * @var int|null
	 * @ORM\Column(type="integer", nullable=true)
	 */
	public $currentRepetitions;

	/**
	 * @var OrderItem|null
	 * @ORM\ManyToOne(targetEntity="EshopOrders\Model\Entities\OrderItem")
	 * @ORM\JoinColumn(name="order_item_id", referencedColumnName="id", onDelete="SET NULL", nullable=true)
	 */
	public $orderItem;

	/**
	 * @ORM\Column(name="description", type="text", nullable=true)
	 */
	public ?string $description;

	/**
	 * @var Collection<OrderSaleInSite>
	 * @ORM\OneToMany(targetEntity="OrderSaleInSite", mappedBy="orderSale", indexBy="site", cascade={"all"}, orphanRemoval=true)
	 */
	public Collection $sites;

	/**
	 * @ORM\Column(name="customer_groups", type="string", nullable=true)
	 */
	protected ?string $customerGroups = null;

	/**
	 * @ORM\Column(name="manufacturers", type="string", nullable=true)
	 */
	protected ?string $manufacturers = null;

	/**
	 * @ORM\Column(name="categories", type="string", nullable=true)
	 */
	protected ?string $categories = null;

	/**
	 * @ORM\Column(name="features", type="string", nullable=true)
	 */
	protected ?string $features = null;

	/**
	 * @ORM\Column(name="products", type="string", nullable=true)
	 */
	protected ?string $products = null;

	public function __construct(string $type, float $amount, float $fromPrice, ?int $maxRepetitions = null)
	{
		$this->setType($type);
		$this->amount             = Strings::formatEntityDecimal($amount);
		$this->fromPrice          = Strings::formatEntityDecimal($fromPrice);
		$this->isActive           = true;
		$this->maxRepetitions     = $maxRepetitions;
		$this->currentRepetitions = $maxRepetitions;
		$this->sites              = new ArrayCollection;
	}

	public function addSite(Site $site): void
	{
		$this->sites->add(new OrderSaleInSite($this, $site));
	}

	public function setType(string $type): self
	{
		if (!in_array($type, array_keys(self::TYPES)))
			throw new \InvalidArgumentException();

		$this->type = $type;

		return $this;
	}

	public function getType(): string { return $this->type; }

	public function getTypeSymbol(): string { return self::TYPES[$this->type]['symbol']; }

	public function setFromPrice(float $fromPrice): self
	{
		$this->fromPrice = Strings::formatEntityDecimal($fromPrice);

		return $this;
	}

	public function getFromPrice(): float { return (float) $this->fromPrice; }

	public function setAmount(float $amount): self
	{
		$this->amount = Strings::formatEntityDecimal($amount);

		return $this;
	}

	public function getAmount(): float { return (float) $this->amount; }

	public function setDateFrom(?DateTime $from = null): self
	{
		$this->dateFrom = $from;

		return $this;
	}

	public function getDateFrom(): ?DateTime { return $this->dateFrom; }

	public function setDateTo(?DateTime $to = null): self
	{
		$this->dateTo = $to;

		return $this;
	}

	public function getDateTo(): ?DateTime { return $this->dateTo; }

	public static function getTypesOptions(): array
	{
		$result = [
			self::TYPE_FIX     => self::TYPES[self::TYPE_FIX]['title'],
			self::TYPE_PERCENT => self::TYPES[self::TYPE_PERCENT]['title'],
		];

		if (EshopSalesConfig::load('allowedTypes.deliveryPrice', false)) {
			$result[self::TYPE_DELIVERY_PRICE] = self::TYPES[self::TYPE_DELIVERY_PRICE]['title'];
		}

		if (EshopSalesConfig::load('allowedTypes.deliveryPriceFirstOrder', false)) {
			$result[self::TYPE_DELIVERY_PRICE_FIRST_ORDER] = self::TYPES[self::TYPE_DELIVERY_PRICE_FIRST_ORDER]['title'];
		}

		if (EshopSalesConfig::load('allowedTypes.randomCategoryProduct', false)) {
			$result[self::TYPE_RANDOM_CAT_PRODUCT] = self::TYPES[self::TYPE_RANDOM_CAT_PRODUCT]['title'];
		}

		if (EshopSalesConfig::load('allowedTypes.product', false)) {
			$result[self::TYPE_PRODUCT] = self::TYPES[self::TYPE_PRODUCT]['title'];
		}

		return $result;
	}

	public function isUsageRepeatable(): bool
	{
		return $this->isUsageInfinitelyRepeatable() || $this->maxRepetitions > 1;
	}

	public function isUsageInfinitelyRepeatable(): bool
	{
		return $this->maxRepetitions === null;
	}

	public function isCurrentRepeatable(): bool
	{
		return $this->isUsageInfinitelyRepeatable() || $this->maxRepetitions > $this->currentRepetitions;
	}

	public function decreaseCurrentRepetitions(): void
	{
		if ($this->isUsageInfinitelyRepeatable()) {
			return;
		}

		if ($this->maxRepetitions !== null && $this->currentRepetitions === null) {
			$this->currentRepetitions = $this->maxRepetitions;
		}

		$this->currentRepetitions--;
		$this->currentRepetitions = max($this->currentRepetitions, 0);
	}

	public function isAutoSale(): bool
	{
		return Validators::isNone($this->code);
	}

	public function getValue(): float
	{
		return (float) $this->amount;
	}

	public function toUsedOrderSale(Order $appliedInOrder, OrderItem $createdByOrderItem = null): UsedOrderSale
	{
		$uos = new UsedOrderSale(
			$this->type, $this->code, $this->getValue(), $this->getFromPrice(), $appliedInOrder,
			$this->dateFrom, $this->dateTo, $this->description
		);

		if ($createdByOrderItem) {
			$uos->createdByOrderItem = $createdByOrderItem;
		}

		/** @var OrderSaleInSite $item */
		foreach ($this->sites->toArray() as $item) {
			$uos->sites->add(new UsedOrderSaleInSite($uos, $item->site));
		}

		return $uos;
	}

	public function setCustomerGroups(?array $groups): void
	{
		$this->customerGroups = $groups ? implode(',', $groups) : null;
	}

	public function getCustomerGroups(): array { return explode(',', (string) $this->customerGroups); }

	public function setManufacturers(?array $manu): void
	{
		$this->manufacturers = $manu ? implode(',', $manu) : null;
	}

	public function getManufacturers(): array { return explode(',', (string) $this->manufacturers); }

	public function setCategories(?array $cats): void
	{
		$this->categories = $cats ? implode(',', $cats) : null;
	}

	public function getCategories(): array { return explode(',', (string) $this->categories); }

	public function setFeatures(?array $fea): void
	{
		$this->features = $fea ? implode(',', $fea) : null;
	}

	public function getFeatures(): array { return explode(',', (string) $this->features); }

	public function setProducts(?array $prods): void
	{
		$this->products = $prods ? implode(',', array_unique($prods)) : null;
	}

	public function getProducts(): array { return explode(',', (string) $this->products); }
}
