<?php declare(strict_types = 1);

namespace EshopSales\Model\Pdf;

use Core\Model\Event\Event;
use Core\Model\Pdf\Pdf;
use EshopSales\Model\Entities\OrderSale;
use Mpdf\Mpdf;
use Mpdf\Output\Destination;
use Nette\Bridges\ApplicationLatte\Template;
use Nette\Utils\Strings;
use Core\Model\Event\EventDispatcher;
use Contributte\Translation\Translator;

class DiscountCouponPdfBuilder
{
	protected OrderSale       $orderSale;
	protected Translator      $translator;
	protected Pdf             $pdfService;
	protected EventDispatcher $eventDispatcher;

	public function __construct(
		OrderSale       $orderSale,
		Translator      $translator,
		Pdf             $pdfService,
		EventDispatcher $eventDispatcher
	)
	{
		$this->orderSale       = $orderSale;
		$this->translator      = $translator;
		$this->pdfService      = $pdfService;
		$this->eventDispatcher = $eventDispatcher;
	}

	public function render(string $destination = Destination::INLINE): string
	{
		$pdf = $this->build();

		return $pdf->Output($this->getFileName(), $destination);
	}

	private function build(): Mpdf
	{
		// css
		$customCssFile  = TEMPLATES_DIR . '/Front/default/EshopSales/_pdf/discountCouponStyle.css';
		$defaultCssFile = __DIR__ . '/../../Model/_pdf/discountCouponStyle.css';

		// latte
		$customTemplate  = TEMPLATES_DIR . '/Front/default/EshopSales/_pdf/discountCoupon.latte';
		$defaultTemplate = __DIR__ . '/../../Model/_pdf/discountCoupon.latte';

		// build template
		$template = $this->pdfService->getTemplateFactory();
		$template->setFile(file_exists($customTemplate) ? $customTemplate : $defaultTemplate);
		$template->orderSale = $this->orderSale;

		// service config
		$pdf = $this->pdfService->newPdf();
		$pdf->WriteHTML(file_get_contents(file_exists($customCssFile) ? $customCssFile : $defaultCssFile) ?: '', 1);

		/** @var Template $template */
		$pdf->WriteHTML($template->renderToString(), 0);

		$pdf->title = $this->translator->translate('eshopSales.discountCoupon.title', null, ['code' => $this->orderSale->code]);

		$event = new Event([
			'template'   => $template,
			'orderSale'  => $this->orderSale,
			'pdfService' => $pdf,
		]);
		$this->eventDispatcher->dispatch($event, self::class . '::build');

		return $pdf;
	}

	private function getTitle(): string
	{
		return $this->translator->translate('eshopSales.discountCoupon.title', null, ['code' => $this->orderSale->code]);
	}

	public function getFileName(): string
	{
		return strtolower(Strings::webalize($this->getTitle()) . '.pdf');
	}
}
