<?php declare(strict_types = 1);

namespace EshopSales\Model\Subscribers;

use Core\Model\Helpers\Arrays;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Statuses;
use EshopSales\AdminModule\Model\OrderSales;
use EshopSales\Model\Entities\OrderSale;
use EshopSales\Model\EshopSalesConfig;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderStatusSubscriber implements EventSubscriberInterface
{
	protected OrderSales $orderSales;
	protected Statuses   $statusesService;

	public function __construct(
		OrderSales $orderSales,
		Statuses   $statusesService
	)
	{
		$this->orderSales      = $orderSales;
		$this->statusesService = $statusesService;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			Order::class . '::changeStatus' => ['orderChangeStatus', 101],
		];
	}

	public function orderChangeStatus(OrderStatusEvent $event): void
	{
		if ($event->status !== OrderStatus::STATUS_IS_PAID) {
			return;
		}

		if (EshopSalesConfig::load('autoGenerateDiscountCoupon.enable', false)) {

			// ---------------- vygenerovani slev. kuponu
			$orderSales = [];
			$orderItems = array_filter($event->order->getOrderItems()->toArray(), static fn(OrderItem $oi) => $oi->getProduct() && $oi->getProduct()->isDiscount);
			/** @var OrderItem $orderItem */
			foreach ($orderItems as $orderItem) {
				$expiration        = (int) EshopSalesConfig::load('autoGenerateDiscountCoupon.expiration');
				$dateFrom          = (new DateTime)->setTime(0, 0);
				$dateTo            = ((new DateTime)->modify("+{$expiration} days"))->setTime(0, 0);
				$amount            = $orderItem->getProduct()->price;
				$isActive          = true;
				$repetitions       = (int) EshopSalesConfig::load('autoGenerateDiscountCoupon.repetitions', 1);
				$bulkCreationCount = $orderItem->getQuantity();

				$orderSales = array_merge($this->orderSales->save([
					'type'                          => OrderSale::TYPE_FIX,
					'dateFrom'                      => $dateFrom,
					'dateTo'                        => $dateTo,
					'amount'                        => (float) $amount,
					'fromPrice'                     => (float) $amount, // od ceny bude stejné jako od hodnoty kuponu
					'isActive'                      => $isActive,
					'code'                          => null,
					'maxRepetitions'                => $repetitions,
					'bulkCreationCount'             => $bulkCreationCount,
					'orderItem'                     => $orderItem,
					'editedId'                      => null,
					'autoGenerateCodeIfCodeIsEmpty' => true,
					'description'                   => null,
				]), $orderSales);
			}

			$afterSendSetExpedition = EshopSalesConfig::load('autoGenerateDiscountCoupon.afterSendSetExpedition', false);
			$afterSendSetFinished   = EshopSalesConfig::load('autoGenerateDiscountCoupon.afterSendSetFinished', false);
			if ($orderSales && $event->order->isPaid && ($afterSendSetExpedition || $afterSendSetFinished)) {
				$isEveryOrderItemDiscount = Arrays::every($event->order->getOrderItems()->toArray(), static fn(OrderItem $oi) => $oi->getProduct() !== null && $oi->getProduct()->isDiscount);
				$isEveryDiscountByEmail   = Arrays::every($orderItems, static fn(OrderItem $oi) => $oi->getProduct() !== null && $oi->getProduct()->getMoreDataValue('discountShipmentMethod') !== 'physical');

				if ($isEveryOrderItemDiscount && $isEveryDiscountByEmail) {
					if ($afterSendSetExpedition && !$event->order->isShipped()) {
						$statusSpedition = $this->statusesService->get(OrderStatus::STATUS_SPEDITION);
						$event->order->getOrderStatuses()->add(new OrderStatus($event->order, $statusSpedition));
					}

					if ($afterSendSetFinished && !$event->order->isFinished()) {
						$statusFinished = $this->statusesService->get(OrderStatus::STATUS_FINISHED);
						$event->order->getOrderStatuses()->add(new OrderStatus($event->order, $statusFinished));
					}
				}
			}


			// ---------------- odeslani slev. kuponu na emaily
			$orderSalesToSend = array_filter($orderSales, static function(OrderSale $orderSale) {
				$sendPhysicalToEmail = EshopSalesConfig::load('autoGenerateDiscountCoupon.sendPhysicalToEmail', false);

				if ($sendPhysicalToEmail) {
					return true;
				}

				$orderItem = $orderSale->orderItem;
				/** @phpstan-ignore-next-line */
				if ($orderItem && $orderItem->getProduct() && $orderItem->getProduct()->getMoreDataValue('discountShipmentMethod') === 'physical') {
					return false;
				}

				return true;
			});
			if ($orderSalesToSend) {
				$this->orderSales->sendDiscountCoupons($orderSalesToSend);
			}
		}
	}

}
