<?php declare(strict_types = 1);

namespace EshopSales\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Doctrine\ORM\QueryBuilder;
use EshopSales\Model\Entities\OrderSale;
use Exception;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Utils\DateTime;

class OrderSales extends BaseFrontEntityService
{
	/** @var array */
	protected $cAutoSales;

	/** @var string */
	protected $entityClass = OrderSale::class;

	/** @var string */
	private const SESSION_SECTION = 'appliedDiscountCouponsSection';

	/** @var string */
	private const KEY_CODES = 'codes';

	/** @var SessionSection */
	protected $sessionSection;

	/**
	 * OrderSales constructor.
	 * @param Session $session
	 */
	public function __construct(Session $session) {
		$this->sessionSection = $session->getSection(self::SESSION_SECTION);
	}

	public function getAutoSales(): array
	{
		if ($this->cAutoSales === null) {
			$today  = new DateTime();
			$result = [];

			foreach ($this->getEr()->createQueryBuilder('os')->where('os.code IS NULL')->andWhere('os.isActive = 1')
				         ->andWhere('os.dateFrom >= :today OR os.dateFrom IS NULL')->andWhere('os.dateTo <= :today OR os.dateTo IS NULL')
				         ->setParameter('today', $today)
				         ->orderBy('os.fromPrice', 'ASC')->getQuery()->getArrayResult() as $row) {
				$row['fromPrice']  = (float) $row['fromPrice'];
				$row['amount']     = (float) $row['amount'];
				$row['typeSymbol'] = OrderSale::TYPES[$row['type']]['symbol'];
				$result[]          = $row;
			}

			$this->cAutoSales = $result;
		}

		return $this->cAutoSales;
	}

	public function getNextSales(float $cartItemsPrice): array
	{
		$result    = [];
		$autoSales = $this->getAutoSales();

		foreach ($autoSales as $k => $sale) {
			if ($cartItemsPrice >= $sale['fromPrice'])
				continue;

			$result[] = $sale;
		}

		return $result;
	}

	/**
	 * @param array $codes
	 * @param float $cartItemsPrice
	 * @return QueryBuilder
	 * @throws Exception
	 */
	private function getQueryBuilderByValidCodes(array $codes, float $cartItemsPrice): QueryBuilder
	{
		$today = new DateTime;
		$qb = $this->getEr()->createQueryBuilder('os');
		$qb->andWhere('os.isActive = 1')
		   ->andWhere('os.dateFrom <= :today OR os.dateFrom IS NULL')
		   ->andWhere('os.dateTo >= :today OR os.dateTo IS NULL')
		   ->andWhere('os.code IN (:codes) OR os.code IS NULL')
		   ->andWhere('os.fromPrice <= :price')
		   ->setParameters([
			   'today' => $today,
			   'codes' => $codes,
			   'price' => $cartItemsPrice
		   ]);

		return $qb;
	}

	/**
	 * @param string $code
	 * @param float $cartItemsPrice
	 * @return bool
	 * @throws Exception
	 */
	public function isValidDiscountCode(string $code, float $cartItemsPrice): bool
	{
		$result = $this->getQueryBuilderByValidCodes([$code], $cartItemsPrice)
					   ->andWhere('os.code IS NOT NULL')
					   ->getQuery()
					   ->getOneOrNullResult();

		return $result !== null;
	}

	/**
	 * @param int $id
	 */
	public function addOrderSaleToCart(int $id): void
	{
		$arr = $this->sessionSection->{self::KEY_CODES};
		$arr[] = $id;
		$arr = array_unique($arr); // Removes duplicate values
		$this->sessionSection->{self::KEY_CODES} = $arr;
	}

	/**
	 * @param OrderSale[] $orderSales
	 */
	private function rewriteCodesInCart(array $orderSales): void
	{
		$this->sessionSection->{self::KEY_CODES} = array_map(static function (OrderSale $sale) {
			return $sale->getId();
		}, $orderSales);
	}

	public function clearCodesInCart(): void
	{
		$this->rewriteCodesInCart([]);
	}

	/**
	 * @param int $id
	 */
	public function removeFromCart(int $id): void
	{
		foreach ($this->sessionSection->{self::KEY_CODES} as $key => $tmpId) {
			if ($tmpId === $id) {
				unset($this->sessionSection->{self::KEY_CODES}[$key]);
				break;
			}
		}
	}

	/**
	 * @param float $cartItemsPrice
	 * @return OrderSale[]
	 * @throws Exception
	 */
	public function getOrderSalesFromCart(float $cartItemsPrice): array
	{
		$storedCodes = [];
		foreach ($this->sessionSection->{self::KEY_CODES} ?? [] as $id) {
			$orderSale = $this->get($id);

			if ($orderSale !== null) {
				$storedCodes[] = $orderSale->code;
			}

		}

		$result = $this->getQueryBuilderByValidCodes($storedCodes, $cartItemsPrice)
					   ->addOrderBy('os.code')
					   ->getQuery()
					   ->getResult();

		$withNullCode = array_filter($result, static function (OrderSale $orderSale): bool {
			return $orderSale->code === null;
		});

		$withNotNullCode = array_filter($result, static function (OrderSale $orderSale): bool {
			return $orderSale->code !== null;
		});

		$orderSales = $withNotNullCode + $withNullCode;

		// rewrite old codes. The original codes may no longer be valid
		$this->rewriteCodesInCart($orderSales);

		return $orderSales;
	}

	/**
	 * @param int $id
	 * @return OrderSale|null
	 */
	public function get(int $id): ?OrderSale
	{
		return $this->getEr()->find($id);
	}

	/**
	 * @param string $code
	 * @return OrderSale|null
	 */
	public function getByCode(string $code): ?OrderSale
	{
		return $this->getEr()->findOneBy(['code' => $code]);
	}
}
