<?php declare(strict_types = 1);

namespace EshopSales\FrontModule\Model\Subscribers;

use Contributte\EventDispatcher\EventSubscriber;
use Doctrine\ORM\EntityManager;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao\Discount;
use EshopOrders\FrontModule\Model\Event\FillDaoEvent;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use EshopSales\FrontModule\Model\OrderSales;
use EshopSales\Model\Entities\OrderSale;
use Exception;
use Nette\Utils\Validators;

class CartSubscriber implements EventSubscriber
{
	/** @var OrderSales */
	protected $orderSales;

	/** @var Carts */
	protected $carts;

	/** @var EntityManager */
	protected $entityManager;

	public function __construct(OrderSales $orderSales, Carts $carts, EntityManager $entityManager)
	{
		$this->orderSales = $orderSales;
		$this->carts = $carts;
		$this->entityManager = $entityManager;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.cartFillDao' => ['cartFillDao', 99],
			'eshopOrders.cartUpdateItem' => ['cartUpdateItem', 99],
			'eshopOrders.orderBeforeSave' => ['orderBeforeSave', 99],
		];
	}

	/**
	 * @param FillDaoEvent $event
	 * @throws Exception
	 */
	public function cartFillDao(FillDaoEvent $event): void
	{
		$cart = $event->cart;

		$cartItemsPrice = $cart->getCartItemsPrice();
		$sales          = $this->orderSales->getOrderSalesFromCart($cartItemsPrice);

		foreach ($sales as $k => $sale) {
			$code = $sale->isAutoSale() ? OrderSale::AUTO_SALE_ID : $sale->code;

			$discount             = new Discount($code, $sale->getAmount(), $sale->getType());
			$discount->typeSymbol = OrderSale::TYPES[$sale->getType()]['symbol'];
			$cart->addDiscount($code, $discount);
		}
	}

	/**
	 * @param UpdatedCartItemEvent $event
	 */
	public function cartUpdateItem(UpdatedCartItemEvent $event): void
	{
		$cartItemsCount = $this->carts->getCurrentCart()->getItemsCount();

		if ($cartItemsCount === 0) {
			$this->orderSales->clearCodesInCart();
		}
	}

	/**
	 * @param OrderEvent $event
	 * @throws Exception
	 */
	public function orderBeforeSave(OrderEvent $event): void
	{
		$currentCartItemsPrice = $this->carts->getCurrentCartItemsPrice();
		$sales = $this->orderSales->getOrderSalesFromCart($currentCartItemsPrice);
		foreach ($sales as $sale) {
			$sale->decreaseCurrentRepetitions();

			if ($sale->currentRepetitions === 0) {
				$this->entityManager->remove($sale);
			} else {
				$this->entityManager->persist($sale);
			}
		}

		$this->orderSales->clearCodesInCart();
	}

}
