<?php declare(strict_types = 1);

namespace EshopSales\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopSales\AdminModule\Model\OrderSales;
use EshopSales\Model\Config;
use EshopSales\Model\Entities\OrderSale;
use Nette\Utils\ArrayHash;
use Nette\Utils\Random;

class OrderSaleForm extends BaseControl
{
	/** @var int @persistent */
	public $id;

	/** @var OrderSale */
	protected $orderSale;

	/** @var OrderSales */
	protected $orderSales;

	/** @var Config */
	protected $config;

	public function __construct(OrderSales $orderSales, Config $config)
	{
		$this->orderSales = $orderSales;
		$this->config = $config;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('code', 'eshopSales.orderSale.code')->setNullable();
		$form->addText('amount', 'eshopSales.orderSale.amount')->setType('number')
			->setAttribute('min', 0)->setAttribute('step', 1)
			->setRequired();
		$form->addSelect('type', 'eshopSales.orderSale.type', OrderSale::getTypesOptions())->setRequired();
		$form->addText('fromPrice', 'eshopSales.orderSale.fromPrice')
			 ->setType('number')
			 ->setAttribute('min', 0)
			 ->setAttribute('step', 1)
			 ->setRequired()
			 ->addRule($form::FLOAT)
			 ->addConditionOn($form['type'], $form::EQUAL, OrderSale::TYPE_FIX)
			 	->addRule($form::MIN, null, $form['amount']);
		$form->addDateTime('dateFrom', 'eshopSales.orderSale.dateFrom');
		$form->addDateTime('dateTo', 'eshopSales.orderSale.dateTo');

		$form->addInteger('repetitions', 'eshopSales.orderSale.repetitions')->setNullable();

		$form->addBool('isActive', 'default.isActive')->setDefaultValue(1);

		$form->addBool('bulkCreation', 'eshopSales.orderSale.bulkCreation.caption')->setRequired();
		$form->addInteger('bulkCreationCount', 'eshopSales.orderSale.bulkCreationCount')
		     ->setNullable()
			 ->addConditionOn($form['bulkCreation'], $form::EQUAL, '1')
			    ->setRequired();

		$form->addHidden('orderSaleId');

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$this->em->beginTransaction();
			$codesConfig = $this->config->get('codes');

			if ($this->id) {
				$bulkCreationCount = 1;
			} else {
				$bulkCreationCount = $values->bulkCreationCount ?? 1;
			}

			for ($i = 0; $i < $bulkCreationCount; $i++) {

				if ($bulkCreationCount > 1) {
					$values->code = Random::generate($codesConfig['length'], $codesConfig['charlist']);
				}

				$sale = $this->id
					? $this->orderSales->get($this->id)
					: new OrderSale($values->type, (float) $values->amount, (float) $values->fromPrice, $values->repetitions);

				$sale->setType($values->type);
				$sale->setDateFrom($values->dateFrom);
				$sale->setDateTo($values->dateTo);
				$sale->setAmount((float) $values->amount);
				$sale->setFromPrice((float) $values->fromPrice);
				$sale->isActive = $values->isActive;
				$sale->code = $values->code;
				$sale->maxRepetitions = $values->repetitions;

				$this->em->persist($sale)->flush();
				if ($bulkCreationCount === 1) {
					$form->addCustomData('orderSaleId', $sale->getId());
				}
			}

			$this->em->commit();
			$this->getPresenter()->flashMessageSuccess('default.saved');

			$this->em->beginTransaction();
		} catch (\Exception $e) {
			$this->em->rollback();
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setOrderSale(int $id): void
	{
		$this->orderSale = $this->orderSales->get($id);

		if (!$this->orderSale)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);

		$s = $this->orderSale;
		$d = [
			'code'        => $s->code,
			'amount'      => $s->getAmount(),
			'fromPrice'   => $s->getFromPrice(),
			'dateFrom'    => $s->getDateFrom(),
			'dateTo'      => $s->getDateTo(),
			'repetitions' => $s->maxRepetitions,
			'isActive'    => (int)$s->isActive,
			'orderSaleId' => $id
		];

		if ($s->getType() && array_key_exists($s->getType(), $this['form']['type']->getItems()))
			$d['type'] = $s->getType();

		$this['form']->setDefaults($d);
		$this->redrawControl('form');
	}
}
