<?php declare(strict_types = 1);

namespace EshopSales\FrontModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\CartHelper;
use EshopSales\FrontModule\Model\CartSales;
use EshopSales\Model\Config;
use Exception;
use Nette\Forms\IControl;
use Nette\Utils\ArrayHash;
use Nette\Http\SessionSection;

class DiscountForm extends BaseControl
{
	/** @var Carts */
	private $cartsService;
	
	/** @var CartHelper */
	protected $cartHelperService;

	/** @var CartSales */
	protected $cartSales;

	/** @var SessionSection */
	protected $sessionSection;

	/** @var Config */
	protected $config;
	
	public function __construct(CartSales $cartSales, Config $config, Carts $carts, CartHelper $cartHelper)
	{
		parent::__construct();

		$this->cartSales = $cartSales;
		$this->config = $config;
		
		$this->cartsService      = $carts;
		$this->cartHelperService = $cartHelper;
	}

	/**
	 * @return BaseForm
	 */
	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('discountCode', 'eshopOrders.orderPage.discountCode')
			 ->setRequired()
			 ->addRule(function (IControl $control) {
			 	$cartItemsPrice = $this->cartsService->getCurrentCartItemsPrice();
			 	return $this->cartSales->isValidDiscountCode($control->getValue(), $cartItemsPrice);
			 }, 'eshopSales.discountForm.discountCodeInvalid');

		$form->addSubmit('submit', 'eshopSales.discountForm.submit');

		$form->onValidate[] = [$this, 'formOnValidate'];
		$form->onSuccess[] = [$this, 'formOnSuccess'];

		return $form;
	}

	/**
	 * @param BaseForm $form
	 * @param ArrayHash $values
	 * @throws Exception
	 */
	public function formOnValidate(BaseForm $form, ArrayHash $values): void
	{
		$cartItemsPrice = $this->cartsService->getCurrentCartItemsPrice();
		if (count($this->cartSales->getOrderSalesFromCart($cartItemsPrice)) === $this->config->get('discountCouponsCanBeRedeemedAtOnce')) {
			$form->addError('eshopSales.discountForm.discountCodesExhausted');
		}

		if ($form->hasErrors() && $this->getParent() !== null) {
			$this->getParent()->redrawControl('discountForm');
		}
	}

	/**
	 * @param BaseForm $form
	 * @param ArrayHash $values
	 */
	public function formOnSuccess(BaseForm $form, ArrayHash $values): void
	{
		$this->cartSales->addDiscountCodeToCart($values->discountCode);
		$this->eventDispatcher->dispatch('eshopSales.discountCodesChanged');
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}
}

