<?php declare(strict_types=1);

namespace EshopSales\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Doctrine\ORM\QueryBuilder;
use EshopSales\Model\Entities\OrderSale;
use Exception;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Utils\DateTime;

class CartSales extends BaseFrontEntityService
{
	/** @var string */
	private const SESSION_SECTION = 'appliedDiscountCouponsSection';

	/** @var string */
	private const KEY_CODES = 'codes';

	/** @var string */
	protected $entityClass = OrderSale::class;

	/** @var SessionSection */
	protected $sessionSection;

	public function __construct(Session $session) {
		$this->sessionSection = $session->getSection(self::SESSION_SECTION);
	}

	/**
	 * @param array $codes
	 * @param float $cartItemsPrice
	 * @return QueryBuilder
	 * @throws Exception
	 */
	private function getQueryBuilderByValidCodes(array $codes, float $cartItemsPrice): QueryBuilder
	{

		$today = new DateTime;
		$qb = $this->getEr()->createQueryBuilder('os');
		$qb->andWhere('os.isActive = 1')
		   ->andWhere('os.dateFrom <= :today OR os.dateFrom IS NULL')
		   ->andWhere('os.dateTo >= :today OR os.dateTo IS NULL')
		   ->andWhere('os.code IN (:codes)')
		   ->andWhere('os.fromPrice <= :price')
		   ->setParameters([
			   'today' => $today,
			   'codes' => $codes,
			   'price' => $cartItemsPrice
		   ]);

		return $qb;
	}

	/**
	 * @param string $code
	 * @param float $cartItemsPrice
	 * @return bool
	 * @throws Exception
	 */
	public function isValidDiscountCode(string $code, float $cartItemsPrice): bool
	{
		return $this->getQueryBuilderByValidCodes([$code], $cartItemsPrice)->getQuery()->getOneOrNullResult() !== null;
	}

	/**
	 * @param string $code
	 */
	public function addDiscountCodeToCart(string $code): void
	{
		$arr = $this->sessionSection->{self::KEY_CODES};
		$arr[] = $code;
		$arr = array_unique($arr); // Removes duplicate values
		$this->sessionSection->{self::KEY_CODES} = $arr;
	}

	/**
	 * @param CartSales[] $orderSales
	 */
	private function rewriteCodesInCart(array $orderSales): void
	{
		$this->sessionSection->{self::KEY_CODES} = array_map(static function (OrderSale $sale) {
			return $sale->code;
		}, $orderSales);
	}

	public function clearCodesInCart(): void
	{
		$this->rewriteCodesInCart([]);
	}

	/**
	 * @param string $code
	 */
	public function removeFromCart(string $code): void
	{
		foreach ($this->sessionSection->{self::KEY_CODES} as $key => $tmpCode) {
			if ($tmpCode === $code) {
				unset($this->sessionSection->{self::KEY_CODES}[$key]);
				break;
			}
		}
	}

	/**
	 * @param float $cartItemsPrice
	 * @return OrderSale[]
	 * @throws Exception
	 */
	public function getOrderSalesFromCart(float $cartItemsPrice): array
	{
		$storedCodes = $this->sessionSection->{self::KEY_CODES} ?? [];
		$orderSales = $this->getQueryBuilderByValidCodes($storedCodes, $cartItemsPrice)->getQuery()->getResult();

		// rewrite old codes. The original codes may no longer be valid
		$this->rewriteCodesInCart($orderSales);

		return $orderSales;
	}

	/**
	 * @param int $id
	 * @return OrderSale|null
	 */
	public function get(int $id): ?OrderSale
	{
		return $this->getEr()->find($id);
	}

}