<?php declare(strict_types = 1);

namespace EshopSales\Model\Entities;

use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;
use \DateTime;

/**
 * @ORM\Table("eshop_sales__order_sale")
 * @ORM\Entity
 * @ORM\EntityListeners({"EshopSales\Model\Listeners\OrderSaleListener"})
 */
class OrderSale
{
	use Identifier;
	use MagicAccessors;

	const TYPE_FIX     = 'fix';
	const TYPE_PERCENT = 'percent';

	const TYPES = [
		self::TYPE_FIX     => [
			'title'  => 'eshopSales.types.' . self::TYPE_FIX,
			'symbol' => '',
		],
		self::TYPE_PERCENT => [
			'title'  => 'eshopSales.types.' . self::TYPE_PERCENT,
			'symbol' => '%',
		],
	];

	/**
	 * @var string
	 * @ORM\Column(name="code", type="string", nullable=true)
	 */
	public $code;

	/**
	 * @var bool
	 * @ORM\Column(name="is_active", type="boolean", nullable=false, options={"default": true})
	 */
	public $isActive;

	/**
	 * @var string
	 * @ORM\Column(name="type", type="string", nullable=false, options={"default":"price"})
	 */
	protected $type;

	/**
	 * @var float
	 * @ORM\Column(name="amount", type="decimal", precision=10, scale=3, nullable=false)
	 */
	protected $amount;

	/**
	 * @var float
	 * @ORM\Column(name="from_price", type="decimal", precision=10, scale=3, nullable=false)
	 */
	protected $fromPrice;

	/**
	 * @var DateTime
	 * @ORM\Column(name="date_from", type="datetime", nullable=true)
	 */
	protected $dateFrom;

	/**
	 * @var DateTime
	 * @ORM\Column(name="date_to", type="datetime", nullable=true)
	 */
	protected $dateTo;

	/**
	 * @var int|null
	 * @ORM\Column(type="integer", nullable=true)
	 */
	public $maxRepetitions;

	/**
	 * @var int|null
	 * @ORM\Column(type="integer", nullable=true)
	 */
	public $currentRepetitions;

	public function __construct(string $type, float $amount, float $fromPrice, ?int $maxRepetitions = null)
	{
		$this->setType($type);
		$this->amount    = $amount;
		$this->fromPrice = $fromPrice;
		$this->isActive  = true;
		$this->maxRepetitions = $maxRepetitions;
		$this->currentRepetitions = $maxRepetitions;
	}

	public function setType(string $type): self
	{
		if (!in_array($type, array_keys(self::TYPES)))
			throw new \InvalidArgumentException();

		$this->type = $type;

		return $this;
	}

	public function getType(): string { return $this->type; }

	public function getTypeSymbol(): string { self::TYPES[$this->type]['symbol']; }

	public function setFromPrice(float $fromPrice): self
	{
		$this->fromPrice = $fromPrice;

		return $this;
	}

	public function getFromPrice(): float { return (float) $this->fromPrice; }

	public function setAmount(float $amount): self
	{
		$this->amount = $amount;

		return $this;
	}

	public function getAmount(): float { return (float) $this->amount; }

	public function setDateFrom(?DateTime $from = null): self
	{
		$this->dateFrom = $from;

		return $this;
	}

	public function getDateFrom(): ?DateTime { return $this->dateFrom; }

	public function setDateTo(?DateTime $to = null): self
	{
		$this->dateTo = $to;

		return $this;
	}

	public function getDateTo(): ?DateTime { return $this->dateTo; }

	public static function getTypesOptions(): array
	{
		$result = [];
		foreach (self::TYPES as $k => $v)
			$result[$k] = $v['title'];

		return $result;
	}

	public function isUsageRepeatable(): bool
	{
		return $this->isUsageInfinitelyRepeatable() || $this->maxRepetitions > 1;
	}

	public function isUsageInfinitelyRepeatable(): bool
	{
		return $this->maxRepetitions === null;
	}

	public function isCurrentRepeatable(): bool
	{
		return $this->isUsageInfinitelyRepeatable() || $this->maxRepetitions > $this->currentRepetitions;
	}

	public function decreaseCurrentRepetitions(): void
	{
		if ($this->isUsageInfinitelyRepeatable()) {
			return;
		}

		$this->currentRepetitions--;
		$this->currentRepetitions = max($this->currentRepetitions, 0);
	}
	
}
