<?php declare(strict_types = 1);

namespace EshopStatistics\AdminModule\Components\Customer;

use Contributte\Datagrid\DataSource\ArrayDataSource;
use Core\AdminModule\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use DateTime;
use EshopOrders\AdminModule\Model\GroupsCustomers;
use EshopOrders\Model\EshopOrdersConfig;
use EshopStatistics\AdminModule\Model\ProductStatistics;
use Nette\Application\Attributes\Persistent;
use Nette\Http\Session;
use Nette\Utils\ArrayHash;

class CustomerStatisticsGrid extends BaseControl
{
	#[Persistent]
	public string $from;

	#[Persistent]
	public string $to;

	#[Persistent]
	public ?string $site = null;

	#[Persistent]
	public ?string $customerGroup = null;

	#[Persistent]
	public ?string $orderType = null;

	public function __construct(
		protected ProductStatistics $productStatistics,
		protected Sites             $sites,
		protected GroupsCustomers   $groupsCustomers,
		protected Session           $session,
	)
	{
		$this->from = (new DateTime)->modify('-1 year')->format('Y-m-d');
		$this->to   = (new DateTime)->format('Y-m-d');
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function handleReset(): void
	{
		$this->from          = (new DateTime)->modify('-1 year')->format('Y-m-d');
		$this->to            = (new DateTime)->format('Y-m-d');
		$this->site          = null;
		$this->customerGroup = null;
		$this->orderType     = null;

		$sessionSection = $this->session->getSection($this['grid']->getSessionSectionName());
		$sessionSection->remove();

		$this->redrawControl('customerStatisticsCard');
		$this['grid']->handleResetFilter();
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentFilterForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$sitesSelect    = ['' => 'eshopStatistics.productStatistics.productsSold.allSites'] + $this->sites->getOptionsForSelect();
		$customerGroups = [
			''      => 'eshopStatistics.productStatistics.customers.allGroups',
			'empty' => 'eshopStatistics.productStatistics.customers.withoutGroup',
		];

		foreach ($this->groupsCustomers->getOptionsForSelect() as $k => $v) {
			$customerGroups[$k] = $v;
		}

		$form->addDatePicker('from', '')
			->setHtmlAttribute('data-type', 'yearMonthDay')
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();') //TODO s naja se nacita nejaky jiny datepicker
			->setDefaultValue($this->from);
		$form->addDatePicker('to', '')
			->setHtmlAttribute('data-type', 'yearMonthDay')
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
			->setDefaultValue($this->to);

		if (count($this->sites->getOptionsForSelect()) > 1) {
			$form->addSelect('site', 'eshopStatistics.customerStatistics.site', $sitesSelect)
				->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
				->setDefaultValue($this->site);
		}

		$form->addSelect('group', 'eshopStatistics.customerStatistics.customerGroup', $customerGroups)
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
			->setDefaultValue($this->customerGroup);

		if (EshopOrdersConfig::load('orderForm.allowEditType')) {
			$form->addSelect('orderType', 'eshopStatistics.customerStatistics.orderType', ['' => ''] + (array) EshopOrdersConfig::load('orderForm.typeList'))
				->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
				->setDefaultValue($this->orderType);
		}

		$form->addSubmit('update', 'default.update');

		$form->onSuccess[] = function(BaseForm $form, ArrayHash $values): void {
			$this->from          = $values->from->format('Y-m-d');
			$this->to            = $values->to->format('Y-m-d');
			$this->site          = $values->site;
			$this->customerGroup = $values->group ? (string) $values->group : null;

			if (EshopOrdersConfig::load('orderForm.allowEditType')) {
				$this->orderType = $values->orderType ?: null;
			}

			$this['grid']->reload();
			$this->redrawControl('customerStatisticsCard');
		};

		return $form;
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$data = $this->productStatistics->getByCustomer($this->from, $this->to, $this->site, $this->customerGroup, [
			'orderType' => $this->orderType,
		]);
		uasort($data, static function($a, $b) {
			return $b['price'] <=> $a['price'];
		});

		$dataSource = new ArrayDataSource($data);

		$grid->setDataSource($dataSource);

		// Columns
		$grid->addColumnLink('id', 'eshopStatistics.productStatistics.customers.name', ':EshopOrders:Admin:Customers:editCustomer', 'name')->setSortable();
		$grid->addColumnText('email', 'eshopStatistics.productStatistics.customers.email')->setSortable();
		$grid->addColumnNumber('count', 'eshopStatistics.productStatistics.customers.count')->setFormat(0, '', ' ')->setSortable();
		$columnPrice = $grid->addColumnNumber('price', 'eshopStatistics.productStatistics.customers.price')->setFormat(2, ',', ' ')->setSortable()
			->addCellAttributes(['class' => 'price border-right-stronger']);

		$columnPrice->setSortableCallback(function($items, $sort) {
			if (isset($sort['price'])) {
				if ($sort['price'] === 'DESC') {
					uasort($items, static function($a, $b) {
						return $b['price'] <=> $a['price'];
					});
				} else {
					uasort($items, static function($a, $b) {
						return $a['price'] <=> $b['price'];
					});
				}
			}

			return $items;
		});

		$grid->setColumnsSummary(['count', 'price']);

		return $grid;
	}

}
