<?php declare(strict_types = 1);

namespace EshopStatistics\AdminModule\Components\Orders;

use Core\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use DateTime;
use DateTimeInterface;
use EshopCatalog\Model\Config;
use EshopStatistics\AdminModule\Model\OrderStatistics;
use EshopStatistics\Model\EshopStatisticsConfig;

class OrderStatisticsDailyGrid extends BaseControl
{
	protected DateTimeInterface $from;
	protected DateTimeInterface $to;

	public function __construct(
		protected OrderStatistics $orderStatistics,
		protected Sites           $sites,
	)
	{
		$this->from = (new DateTime)->modify('-1 year');
		$this->to   = (new DateTime);
	}

	public function render(DateTime $month): void
	{
		$month->modify('first day of this month');
		$this->from = clone $month;
		$month->modify('last day of this month');
		$this->to = clone $month;

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setTemplateFile(__DIR__ . '/../StatisticsGridHeader_grid.latte');

		$qb = $this->orderStatistics->getDailyByShop($this->from, $this->to);
		$grid->setDataSource($qb);
		$grid->setPagination(false);

		$summaryCols = [];

		// Columns
		$grid->addColumnDateTime('date', 'eshopStatistics.orderStatistics.monthlyOrders.date')
			->setFormat('d.m.Y')
			->addCellAttributes(['class' => 'border-right-stronger'])
			->setTemplate(null, ['supHeaderGroup' => 'date']);
		$grid->addColumnNumber('orders', 'eshopStatistics.orderStatistics.monthlyOrders.ordersCount')
			->setFormat(0, '', ' ')
			->setTemplate(null, [
				'supHeaderText'  => $this->translator->translate('eshopStatistics.orderStatistics.monthlyOrders.total'),
				'supHeaderGroup' => 'total',
			]);
		$summaryCols[] = 'orders';

		if (Config::load('enablePurchasePrice')) {
			$grid->addColumnNumber('ordersProfit', 'eshopStatistics.orderStatistics.monthlyOrders.profit')
				->setFormat(2, ',', ' ')
				->addCellAttributes(['class' => 'price'])
				->setTemplate(null, ['supHeaderGroup' => 'total']);
			$summaryCols[] = 'ordersProfit';
		}

		$moreSumColumns = (array) EshopStatisticsConfig::load('ordersStatisticGrid.sumCurrencies') ?: [];

		$grid->addColumnNumber('ordersPrice', 'eshopStatistics.orderStatistics.monthlyOrders.ordersPrice')
			->setFormat(2, ',', ' ')
			->addCellAttributes(['class' => 'price'])
			->setTemplate(null, ['supHeaderGroup' => 'total']);
		$summaryCols[] = 'ordersPrice';

		$grid->addColumnNumber('ordersPriceCZK', 'eshopStatistics.orderStatistics.monthlyOrders.priceCZK')
			->setFormat(2, ',', ' ')
			->addCellAttributes(['class' => 'price'])
			->setTemplate(null, ['supHeaderGroup' => 'total']);
		$summaryCols[] = 'ordersPriceCZK';

		$grid->addColumnNumber('ordersPriceEUR', 'eshopStatistics.orderStatistics.monthlyOrders.priceEUR')
			->setFormat(2, ',', ' ')
			->addCellAttributes(['class' => 'price' . (empty($moreSumColumns) ? ' border-right-stronger' : '')])
			->setTemplate(null, ['supHeaderGroup' => 'total']);
		$summaryCols[] = 'ordersPriceEUR';

		$moreSumColumnsCount = count($moreSumColumns);
		foreach ($moreSumColumns as $k => $curr) {
			$grid->addColumnNumber('ordersPrice' . $curr, $this->t('eshopStatistics.orderStatistics.monthlyOrders.priceX', ['curr' => $curr]))
				->setFormat(2, ',', ' ')
				->setTranslatableHeader(false)
				->addCellAttributes(['class' => 'price' . ($k === $moreSumColumnsCount - 1 ? ' border-right-stronger' : '')])
				->setTemplate(null, ['supHeaderGroup' => 'total']);
			$summaryCols[] = 'ordersPrice' . $curr;
		}

		$sites = $this->sites->getSites(false);
		krsort($sites);
		foreach ($sites as $site) {
			$domain = $site->getDomains()[$this->translator->getLocale()] ?? $site->getDefaultDomain();

			if (!$domain) {
				continue;
			}

			$siteName  = $domain->siteName;
			$siteIdent = $site->getIdent();

			$moreSiteColumns = (array) EshopStatisticsConfig::load('ordersStatisticGrid.shopCurrencies.' . $siteIdent) ?: [];

			$grid->addColumnNumber('orders-' . $siteIdent, 'eshopStatistics.orderStatistics.monthlyOrders.ordersCount')
				->setFormat(0, '', ' ')
				->setTemplate(null, ['supHeaderText' => $siteName, 'supHeaderGroup' => $siteIdent]);
			$summaryCols[] = 'orders-' . $siteIdent;

			if (Config::load('enablePurchasePrice')) {
				$grid->addColumnNumber('ordersProfit-' . $siteIdent, 'eshopStatistics.orderStatistics.monthlyOrders.profit')
					->setFormat(2, ',', ' ')
					->addCellAttributes(['class' => 'price'])
					->setTemplate(null, ['supHeaderGroup' => $siteIdent]);
				$summaryCols[] = 'ordersProfit-' . $siteIdent;
			}

			$grid->addColumnNumber('ordersPrice-' . $siteIdent, 'eshopStatistics.orderStatistics.monthlyOrders.ordersPrice')
				->setFormat(2, ',', ' ')
				->addCellAttributes(['class' => 'price'])
				->setTemplate(null, ['supHeaderGroup' => $siteIdent]);
			$summaryCols[] = 'ordersPrice-' . $siteIdent;

			$grid->addColumnNumber('ordersPrice-' . $siteIdent . '-CZK', 'eshopStatistics.orderStatistics.monthlyOrders.priceCZK')
				->setFormat(2, ',', ' ')
				->addCellAttributes(['class' => 'price'])
				->setTemplate(null, ['supHeaderGroup' => $siteIdent]);
			$summaryCols[] = 'ordersPrice-' . $siteIdent . '-CZK';

			$grid->addColumnNumber('ordersPrice-' . $siteIdent . '-EUR', 'eshopStatistics.orderStatistics.monthlyOrders.priceEUR')
				->setFormat(2, ',', ' ')
				->addCellAttributes(['class' => 'price' . (empty($moreSiteColumns) ? ' border-right-stronger' : '')])
				->setTemplate(null, ['supHeaderGroup' => $siteIdent]);
			$summaryCols[] = 'ordersPrice-' . $siteIdent . '-EUR';

			$moreSiteColumnsCount = count($moreSiteColumns);
			foreach ($moreSiteColumns as $k => $curr) {
				$grid->addColumnNumber('ordersPrice-' . $siteIdent . '-' . $curr, $this->t('eshopStatistics.orderStatistics.monthlyOrders.priceX', ['curr' => $curr]))
					->setFormat(2, ',', ' ')
					->setTranslatableHeader(false)
					->addCellAttributes(['class' => 'price' . ($k === $moreSiteColumnsCount - 1 ? ' border-right-stronger' : '')])
					->setTemplate(null, ['supHeaderGroup' => $siteIdent]);
				$summaryCols[] = 'ordersPrice-' . $siteIdent . '-' . $curr;
			}
		}

		$grid->setColumnsSummary($summaryCols);

		return $grid;
	}
}
