<?php declare(strict_types = 1);

namespace EshopStatistics\AdminModule\Components\Product;

use Contributte\Datagrid\DataSource\ArrayDataSource;
use Core\AdminModule\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use DateTime;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\AdminModule\Model\Suppliers;
use EshopStatistics\AdminModule\Model\ProductStatistics;
use EshopStatistics\Model\EshopStatisticsConfig;
use Nette\Application\Attributes\Persistent;
use Nette\Http\Session;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;
use Nette\Utils\Validators;

class ProductStatisticsGrid extends BaseControl
{
	#[Persistent]
	public string $from;

	#[Persistent]
	public string $to;

	#[Persistent]
	public ?string $site = null;

	#[Persistent]
	public ?int $manufacturer = null;

	#[Persistent]
	public ?int $supplier = null;

	#[Persistent]
	public ?string $features = null;

	protected ?array $cData = null;

	public function __construct(
		protected ProductStatistics $productStatistics,
		protected Sites             $sites,
		protected Manufacturers     $manufacturers,
		protected Suppliers         $suppliers,
		protected Session           $session,
	)
	{
		$this->from = (new DateTime)->modify('-1 year')->format('Y-m-d');
		$this->to   = (new DateTime)->format('Y-m-d');
	}

	public function render(): void
	{
		$this->template->lastUpdate     = (new \Nette\Utils\DateTime())->setTimestamp((int) $this->getData()['lastUpdate']);
		$this->template->featureColumns = $this->getData()['featureColumns'];
		$this->template->render($this->getTemplateFile());
	}

	public function handleReset(): void
	{
		$this->from         = (new DateTime)->modify('-1 year')->format('Y-m-d');
		$this->to           = (new DateTime)->format('Y-m-d');
		$this->site         = null;
		$this->manufacturer = null;
		$this->supplier     = null;
		$this->features     = null;

		$sessionSection = $this->session->getSection($this['grid']->getSessionSectionName());
		$sessionSection->remove();

		$this->redrawControl('productStatisticsCard');
		$this['grid']->handleResetFilter();
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentFilterForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$sitesSelect   = ['' => 'eshopStatistics.productStatistics.productsSold.allSites'] + $this->sites->getOptionsForSelect();
		$manufacturers = ['' => 'eshopStatistics.productStatistics.productsSold.allManufacturers'];
		foreach ($this->manufacturers->getOptionsForSelect() as $k => $v) {
			$manufacturers[$k] = $v;
		}

		$suppliers = ['' => 'eshopStatistics.productStatistics.productsSold.allSuppliers'];
		foreach ($this->suppliers->getOptionsForSelect() as $k => $v) {
			$suppliers[$k] = $v;
		}

		$form->addDatePicker('from', '')
			->setHtmlAttribute('data-type', 'yearMonthDay')
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();') //TODO s naja se nacita nejaky jiny datepicker
			->setDefaultValue($this->from);
		$form->addDatePicker('to', '')
			->setHtmlAttribute('data-type', 'yearMonthDay')
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
			->setDefaultValue($this->to);
		$form->addSelect('site', '', $sitesSelect)
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
			->setDefaultValue($this->site);
		$form->addSelect('manufacturer', '', $manufacturers)
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
			->setDefaultValue($this->manufacturer);
		$form->addSelect('supplier', '', $suppliers)
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
			->setDefaultValue($this->supplier);

		if ($this->getData()['featureColumns']) {
			$defaults = $this->features ? Json::decode($this->features, Json::FORCE_ARRAY) : [];

			foreach ($this->getData()['featureColumns'] as $featureId => $featureName) {
				$form->addSelect('feature' . $featureId, $featureName, ['' => ''] + $this->getData()['featureValues'][$featureId])
					->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
					->setDefaultValue($defaults[$featureId] ?? '');
			}
		}

		$form->addSubmit('update', 'default.update');

		$form->onSuccess[] = function(BaseForm $form, ArrayHash $values): void {
			$this->from         = $values->from->format('Y-m-d');
			$this->to           = $values->to->format('Y-m-d');
			$this->site         = $values->site;
			$this->manufacturer = Validators::isNone($values->manufacturer) ? null : ((int) $values->manufacturer);
			$this->supplier     = Validators::isNone($values->supplier) ? null : ((int) $values->supplier);

			$features = [];
			foreach ($this->getData()['featureColumns'] as $id => $v) {
				if ($values->{'feature' . $id}) {
					$features[$id] = $values->{'feature' . $id};
				}
			}

			$this->features = $features ? Json::encode($features) : null;

			$this->cData = null;
			$this->getData();
			$this['grid']->reload();
			$this->redrawControl('productStatisticsCard');
		};

		return $form;
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$dataSource = new ArrayDataSource($this->getData()['data']);
		$grid->setDataSource($dataSource);

		// Columns
		$grid->addColumnText('id', 'eshopStatistics.productStatistics.productsSold.id')
			->setRenderer(static fn(array $row): int => (int) $row['id'])
			->setAlign('right')
			->addCellAttributes(['class' => 'w1nw'])
			->setFilterText();
		$grid->addColumnLink('name', 'eshopStatistics.productStatistics.productsSold.name', ':EshopCatalog:Admin:Products:edit')
			->setSortable()
			->setFilterText();
		$grid->addColumnText('code', 'eshopStatistics.productStatistics.productsSold.code')
			->setSortable()
			->addCellAttributes(['class' => 'border-right-stronger'])
			->setFilterText();

		foreach ($this->getData()['featureColumns'] as $featureId => $featureName) {
			$grid->addColumnText('feature' . $featureId, $featureName)
				->setTranslatableHeader(false)
				->setRenderer(function($row) use ($featureId) {
					return $this->getData()['productsData'][$row['id']]['features'][$featureId] ?? '';
				});
		}

		$grid->addColumnNumber('inOrdersCount', 'eshopStatistics.productStatistics.productsSold.inOrdersCount')
			->setFormat(0, '', ' ')
			->setSortable();
		$grid->addColumnNumber('count', 'eshopStatistics.productStatistics.productsSold.count')
			->setFormat(0, '', ' ')
			->setSortable();
		$grid->addColumnNumber('price', 'eshopStatistics.orderStatistics.monthlyOrders.ordersPrice')
			->setSortable()
			->setSortableCallback(function(array $data, array $sort) {
				$column    = key($sort);
				$direction = $sort[$column];

				usort($data, static function($a, $b) use ($column, $direction) {
					if ($a[$column] === $b[$column]) {
						return 0;
					}

					return ($a[$column] < $b[$column] ? -1 : 1) * ($direction === 'ASC' ? 1 : -1);
				});

				return $data;
			})
			->setRenderer(fn(array $row): string => number_format($row['price'], 2, ',', ' '))
			->addCellAttributes(['class' => 'price border-right-stronger']);

		$grid->setColumnsSummary(['inOrdersCount', 'count', 'price'])->setRenderer(function($val, $col) {
			if ($col === 'price') {
				return number_format($val, 2, ',', ' ');
			}

			return (string) $val;
		});

		$grid->getColumn('inOrdersCount')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('count')->getElementPrototype('th')->addClass('w1nw');

		// Export
		$grid->addExportCsv('CSV Export', 'products.csv', 'utf-8');

		return $grid;
	}

	protected function getData(): array
	{
		if ($this->cData === null) {
			$this->cData                   = $this->productStatistics->getByProduct($this->from, $this->to, $this->site, $this->manufacturer, $this->supplier);
			$this->cData['featureColumns'] = [];
			$this->cData['featureValues']  = [];
			$this->cData['productsData']   = [];

			$featuresIds     = EshopStatisticsConfig::loadArray('productStatisticGrid.featuresInGrid') ?: [];
			$allowedFeatures = [];
			$allowedProducts = [];

			foreach ($this->features ? Json::decode($this->features, Json::FORCE_ARRAY) : [] as $featureId => $featureValue) {
				$allowedFeatures[] = $featureValue;
			}

			if ($featuresIds) {
				$productsIds = [];
				foreach ($this->cData['data'] as $data) {
					if (is_numeric($data['id'])) {
						$productsIds[] = $data['id'];
					}
				}

				if ($productsIds) {
					foreach (array_chunk(array_unique($productsIds), 200) as $chunk) {
						foreach ($this->em->getConnection()->executeQuery("SELECT fp.id_product as product, fp.id_feature_value as valueId, fv.feature_id as featureId, ft.name as featureName, fvt.name as valueName
							FROM eshop_catalog__feature_product fp
							INNER JOIN eshop_catalog__feature_value fv ON fp.id_feature_value = fv.id AND fv.feature_id IN (" . implode(',', $featuresIds) . ")
							INNER JOIN eshop_catalog__feature_texts ft ON fv.feature_id = ft.id AND ft.lang = '" . $this->translator->getLocale() . "'
							INNER JOIN eshop_catalog__feature_value_texts fvt ON fv.id = fvt.id AND fvt.lang = '" . $this->translator->getLocale() . "'
							WHERE fp.id_product IN (" . implode(',', $chunk) . ")")->iterateAssociative() as $row) {
							$this->cData['productsData'][$row['product']]['features'][$row['featureId']] = $row['valueName'];
							$this->cData['featureColumns'][$row['featureId']]                            = $row['featureName'];
							$this->cData['featureValues'][$row['featureId']][$row['valueId']]            = $row['valueName'];

							if (in_array($row['valueId'], $allowedFeatures)) {
								$allowedProducts[] = $row['product'];
							}
						}
					}
				}
			}

			foreach ($this->cData['featureValues'] as $k => &$v) {
				asort($v, SORT_NATURAL);
			}

			if ($allowedFeatures) {
				$this->cData['data'] = array_intersect_key($this->cData['data'], array_flip($allowedProducts));
			}

			uasort($this->cData['data'], static function($a, $b) {
				return $b['price'] <=> $a['price'];
			});
		}

		return $this->cData;
	}
}
