<?php declare(strict_types = 1);

namespace EshopStatistics\AdminModule\Model;

use Contributte\Translation\Translator;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\Query\Parameter;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\Statistics\BaseStatistics;
use EshopProductionWarehouse\Model\Entities\WarehouseOrder;
use EshopStatistics\Model\EshopStatisticsConfig;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Throwable;

class ProductStatistics extends BaseEntityService
{
	public const cacheNamespace = 'eshopStatistics';

	public function __construct(
		protected Translator     $translator,
		protected BaseStatistics $baseStatistics,
		Storage                  $storage
	)
	{
		$this->cache = new Cache($storage, self::cacheNamespace);
	}

	// -------------------- statistiky prodanych produktu - pocet a cena ----------------------------------------------

	/**
	 *
	 * @return array{lastUpdate: int, data: array}
	 * @throws Throwable
	 */
	public function getByProduct(
		string  $from,
		string  $to,
		?string $siteIdent = null,
		?int    $manufacturerId = null,
		?int    $supplier = null
	): array
	{
		$withoutVat = (bool) EshopStatisticsConfig::loadScalar('priceWithoutVat');

		/** @var DateTime $fromDate */
		$fromDate = $from !== '' && $from !== '0' ? new DateTime($from) : (new DateTime)->modify('-1 year');
		$toDate   = $to !== '' && $to !== '0' ? new DateTime($to) : new DateTime;

		$fromDate->setTime(0, 0);
		$toDate->setTime(23, 59, 59);

		$key = 'productStatistics_' . $fromDate->getTimestamp() . '_' . $toDate->getTimestamp() . '_site' . $siteIdent . '_m' . $manufacturerId . '_s' . $supplier;

		return $this->cache->load($key, function(&$dep) use ($key, $fromDate, $toDate, $siteIdent, $manufacturerId, $supplier, $withoutVat) {
			$dep = [
				Cache::Expire => '15 minutes',
			];

			FileSystem::createDir(TMP_DIR . '/eshopStatistics');

			$qb        = $this->em->createQueryBuilder();
			$baseQuery = $this->baseStatistics->getBaseOrdersQB($fromDate, $toDate, $siteIdent);
			$query     = $this->em->createQueryBuilder()
				->select('oi.id, IDENTITY(oi.product) as product, oi.code1, oi.price, oi.quantity, oi.vatRate, oit.name, ois.value as saleValue, ois.type as saleType, IDENTITY(oi.order) as orderId')
				->from(OrderItem::class, ' oi')
				->innerJoin('oi.orderItemTexts', 'oit')
				->leftJoin('oi.sales', 'ois')
				->setParameters($baseQuery->getParameters())
				->where($qb->expr()->in('oi.order', $baseQuery->select('o.id')->getDQL()));

			if ($manufacturerId) {
				$query->innerJoin('oi.product', 'oip', Join::WITH, 'oip.manufacturer = :manu')
					->setParameter('manu', $manufacturerId);
			}

			if ($supplier) {
				if (!Arrays::contains($query->getAllAliases(), 'oip')) {
					$query->innerJoin('oi.product', 'oip');
				}

				$query->innerJoin('oip.suppliers', 'oips', Join::WITH, 'oips.supplier = :supplier')
					->setParameter('supplier', $supplier);
			}

			if (class_exists('EshopProductionWarehouse\Model\Entities\WarehouseOrder')) {
				$query->leftJoin(WarehouseOrder::class, 'wo', Join::WITH, 'wo.order = oi.order')
					->andWhere('wo IS NULL OR wo.countInStatistics = 1');
			}

			$result = [];
			foreach ($query->getQuery()->getScalarResult() as $row) {
				$key = $row['product'] ?? $row['name'] . '_' . $row['code1'];
				if (!array_key_exists($key, $result)) {
					$result[$key] = $this->createRowByProduct((string) $key);
				}

				$row['price']    = (float) $row['price'];
				$row['quantity'] = (int) $row['quantity'];
				$row['vatRate']  = (int) $row['vatRate'];

				$sales = $row['saleValue'] ? [OrderHelper::calculateSaleValue($row['saleType'], $row['saleValue'], $row['quantity'], $row['price'])] : [];

				$result[$key]['name']                      = $row['name'];
				$result[$key]['code']                      = $row['code1'];
				$result[$key]['count']                     += (int) $row['quantity'];
				$result[$key]['price']                     += ($withoutVat
					? OrderHelper::getItemPriceTotalWithoutVat($row['price'], $row['quantity'], $row['vatRate'], $sales)
					: OrderHelper::getItemPriceTotal($row['price'], $row['quantity'], $sales));
				$result[$key]['inOrders'][$row['orderId']] = $row['orderId'];
			}

			foreach ($result as $k => $v) {
				$result[$k]['inOrdersCount'] = count($v['inOrders']);
			}

			$lastUpdate = time();

			file_put_contents(TMP_DIR . '/eshopStatistics/' . $key . '.json', $lastUpdate);

			return [
				'lastUpdate' => $lastUpdate,
				'data'       => $result,
			];
		});
	}

	protected function createRowByProduct(string $productId): array
	{
		return [
			'id'            => $productId,
			'name'          => '',
			'count'         => 0,
			'price'         => 0,
			'inOrders'      => [],
			'inOrdersCount' => 0,
		];
	}

	// -------------------- statistiky odberatelu - podle poctu/ceny objednavek ----------------------------------------------

	public function getByCustomer(string $from, string $to, ?string $siteIdent = null, ?string $customerGroup = null, array $filters = []): array
	{
		$withoutVat = EshopStatisticsConfig::load('priceWithoutVat', false);
		$fromDate   = $from !== '' && $from !== '0' ? new DateTime($from) : (new DateTime)->modify('-1 year');
		$toDate     = $to !== '' && $to !== '0' ? new DateTime($to) : new DateTime;

		$fromDate->setTime(0, 0);
		$toDate->setTime(23, 59, 59);

		$customers = [];
		$orders    = [];

		// Zakladni data o objednavce a uzivatele
		$query = $this->baseStatistics->getBaseOrdersQB($fromDate, $toDate, $siteIdent);
		if ($customerGroup) {
			$query->join('o.customer', 'c');
			if ($customerGroup === 'empty') {
				$query->andWhere('c.groupCustomers IS NULL');
			} else {
				$query->andWhere('c.groupCustomers = :customerGroup')
					->setParameter('customerGroup', $customerGroup);
			}
		}

		if (isset($filters['orderType'])) {
			$query->andWhere('o.type = :orderType')
				->setParameter('orderType', $filters['orderType']);
		}

		foreach ($query->getQuery()->getScalarResult() as $row) {
			$row['items']     = [];
			$row['discounts'] = [];

			$orders[$row['id']]          = $row;
			$customers[$row['customer']] = [];
		}

		// Sleva objednavky
		foreach ($this->baseStatistics->getSales(array_keys($orders)) as $id => $sales) {
			$orders[$id]['discounts'] = $sales;
		}

		// Produkty
		foreach ($this->baseStatistics->getOrderItems(array_keys($orders)) as $id => $orderItems) {
			$orders[$id]['items'] = $orderItems;
		}

		// Zakaznik
		foreach (array_chunk(array_keys($customers), 900) as $chunk) {
			foreach ($this->em->createQueryBuilder()->select('c.id, cInv.company, cInv.firstName, cInv.lastName, cInv.email')
				         ->from(Customer::class, 'c')
				         ->innerJoin('c.addressInvoice', 'cInv')
				         ->where('c.id IN (:ids)')
				         ->setParameters(new ArrayCollection([new Parameter('ids', $chunk)]))
				         ->getQuery()->getScalarResult() as $row) {
				$customers[$row['id']] = [
					'id'    => $row['id'],
					'name'  => trim(implode(' ', [$row['company'], $row['firstName'], $row['lastName']])),
					'email' => $row['email'],
				];
			}
		}

		$result = [];
		foreach ($orders as $order) {
			if (!$order['customer']) {
				continue;
			}

			/** @var array{id: int, name: ?string, email: ?string} $customer */
			$customer = $customers[$order['customer']];
			$key      = 'k_' . $order['customer'];

			if (!array_key_exists($key, $result)) {
				$result[$key] = $this->createRowByCustomer($customer['id'], $customer['email'], $customer['name']);
			}

			$result[$key]['count']++;
			foreach ($order['items'] as $item) {
				$sales                 = $item['saleValue'] ? [OrderHelper::calculateSaleValue($item['saleType'], $item['saleValue'], $item['quantity'], $item['price'])] : [];
				$result[$key]['price'] += $withoutVat
					? OrderHelper::getItemPriceTotalWithoutVat($item['price'], $item['quantity'], $item['vatRate'], $sales)
					: OrderHelper::getItemPriceTotal($item['price'], $item['quantity'], $sales);
			}

			$orders[$order['id']] = null;
		}

		return $result;
	}

	protected function createRowByCustomer(int $customerId, ?string $customerEmail, ?string $customerName): array
	{
		return [
			'id'    => $customerId,
			'email' => $customerEmail,
			'name'  => $customerName,
			'count' => 0,
			'price' => 0,
		];
	}
}
