<?php declare(strict_types = 1);

namespace EshopStatistics\AdminModule\Components\Product;

use Core\AdminModule\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use DateTime;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\AdminModule\Model\Suppliers;
use EshopCatalog\FrontModule\Model\Categories as FrontCategories;
use EshopStatistics\AdminModule\Model\ProductStatistics;
use EshopStatistics\Model\EshopStatisticsConfig;
use Nette\Utils\ArrayHash;
use Nette\Utils\Html;
use Nette\Utils\Json;
use Nette\Utils\Validators;
use Ublaboo\DataGrid\DataSource\ArrayDataSource;

class ProductStatisticsGrid extends BaseControl
{
	protected ProductStatistics $productStatistics;
	protected Sites             $sites;
	protected Manufacturers     $manufacturers;
	protected Suppliers         $suppliers;
	protected FrontCategories   $frontCategories;

	/** @var string|null @persistent */
	public ?string $from = null;

	/** @var string|null @persistent */
	public ?string $to = null;

	/** @var string|null @persistent */
	public ?string $site = null;

	/** @var int|null @persistent */
	public ?int $manufacturer = null;

	/** @var int|null @persistent */
	public ?int $supplier = null;

	/** @var string|null @persistent */
	public ?string $features = null;

	protected ?array $cData = null;

	public function __construct(
		ProductStatistics $productStatistics,
		Sites             $sites,
		Manufacturers     $manufacturers,
		Suppliers         $suppliers,
		FrontCategories   $frontCategories
	)
	{
		$this->productStatistics = $productStatistics;
		$this->sites             = $sites;
		$this->manufacturers     = $manufacturers;
		$this->suppliers         = $suppliers;
		$this->frontCategories   = $frontCategories;

		$this->from = (new DateTime)->modify('-1 year')->format('Y-m-d');
		$this->to   = (new DateTime)->format('Y-m-d');
	}

	public function render(): void
	{
		$this->template->lastUpdate     = (new \Nette\Utils\DateTime())->setTimestamp((int) $this->getData()['lastUpdate']);
		$this->template->featureColumns = $this->getData()['featureColumns'];
		$this->template->render($this->getTemplateFile());
	}

	public function handleReset(): void
	{
		$this->from         = (new DateTime)->modify('-1 year')->format('Y-m-d');
		$this->to           = (new DateTime)->format('Y-m-d');
		$this->site         = null;
		$this->manufacturer = null;
		$this->supplier     = null;
		$this->features     = null;

		$this['grid']->getSessionData()->remove();
		$this->redrawControl('productStatisticsCard');
		$this['grid']->handleResetFilter();
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentFilterForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$sitesSelect   = ['' => 'eshopStatistics.productStatistics.productsSold.allSites'] + $this->sites->getOptionsForSelect();
		$manufacturers = ['' => 'eshopStatistics.productStatistics.productsSold.allManufacturers'];
		foreach ($this->manufacturers->getOptionsForSelect() as $k => $v) {
			$manufacturers[$k] = $v;
		}

		$suppliers = ['' => 'eshopStatistics.productStatistics.productsSold.allSuppliers'];
		foreach ($this->suppliers->getOptionsForSelect() as $k => $v) {
			$suppliers[$k] = $v;
		}

		$form->addDatePicker('from', '')
			->setHtmlAttribute('data-type', 'yearMonthDay')
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();') //TODO s naja se nacita nejaky jiny datepicker
			->setDefaultValue($this->from);
		$form->addDatePicker('to', '')
			->setHtmlAttribute('data-type', 'yearMonthDay')
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
			->setDefaultValue($this->to);
		$form->addSelect('site', '', $sitesSelect)
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
			->setDefaultValue($this->site);
		$form->addSelect('manufacturer', '', $manufacturers)
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
			->setDefaultValue($this->manufacturer);
		$form->addSelect('supplier', '', $suppliers)
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
			->setDefaultValue($this->supplier);

		if ($this->getData()['featureColumns']) {
			$defaults = $this->features ? Json::decode($this->features, Json::FORCE_ARRAY) : [];

			foreach ($this->getData()['featureColumns'] as $featureId => $featureName) {
				$form->addSelect('feature' . $featureId, $featureName, ['' => ''] + $this->getData()['featureValues'][$featureId])
					->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
					->setDefaultValue($defaults[$featureId] ?? '');
			}
		}

		$form->addSubmit('update', 'default.update');

		$form->onSuccess[] = function(BaseForm $form, ArrayHash $values) {
			$this->from         = $values->from->format('Y-m-d');
			$this->to           = $values->to->format('Y-m-d');
			$this->site         = $values->site;
			$this->manufacturer = Validators::isNone($values->manufacturer) ? null : ((int) $values->manufacturer);
			$this->supplier     = Validators::isNone($values->supplier) ? null : ((int) $values->supplier);

			$features = [];
			foreach ($this->getData()['featureColumns'] as $id => $v) {
				if ($values->{'feature' . $id}) {
					$features[$id] = $values->{'feature' . $id};
				}
			}

			$this->features = $features ? Json::encode($features) : null;

			$this->cData = null;
			$this->getData();
			$this['grid']->reload();
			$this->redrawControl('productStatisticsCard');
		};

		return $form;
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$dataSource = new ArrayDataSource($this->getData()['data']);
		$grid->setDataSource($dataSource);
		$sitesCount = count($this->sites->getOptionsForSelect());

		// Columns
		$grid->addColumnText('id', 'eshopStatistics.productStatistics.productsSold.id')
			->setRenderer(static function(array $row) {
				return (int) $row['id'];
			})
			->setAlign('right')
			->addCellAttributes(['class' => 'w1nw'])
			->setFilterText();
		$grid->addColumnLink('name', 'eshopStatistics.productStatistics.productsSold.name', ':EshopCatalog:Admin:Products:edit')
			->setSortable()
			->setFilterText();
		$grid->addColumnText('code', 'eshopStatistics.productStatistics.productsSold.code')
			->setSortable()
			->addCellAttributes(['class' => !$this->site ? 'border-right-stronger' : ''])
			->setFilterText();

		$grid->addColumnText('mainCategory', 'eshopStatistics.productStatistics.productsSold.mainCategory')
			->setRenderer(function(array $row) use ($sitesCount) {
				$html = Html::el();
				foreach ($row['mainCategory'] ?? [] as $site => $category) {
					if ($sitesCount > 1 && !$this->site) {
						$html->addHtml(Html::el('div')->setText($site . ' > ' . $category));
					} else {
						$html->addHtml(Html::el('div')->setText($category));
					}
				}

				return $html;
			})
			->addCellAttributes(['class' => 'border-right-stronger']);

		foreach ($this->getData()['featureColumns'] as $featureId => $featureName) {
			$grid->addColumnText('feature' . $featureId, $featureName)
				->setTranslatableHeader(false)
				->setRenderer(function($row) use ($featureId) {
					return $this->getData()['productsData'][$row['id']]['features'][$featureId] ?? '';
				});
		}

		$grid->addColumnNumber('inOrdersCount', 'eshopStatistics.productStatistics.productsSold.inOrdersCount')
			->setFormat(0, '', ' ')
			->setSortable();
		$grid->addColumnNumber('count', 'eshopStatistics.productStatistics.productsSold.count')
			->setFormat(0, '', ' ')
			->setSortable();
		$grid->addColumnNumber('price', 'eshopStatistics.orderStatistics.monthlyOrders.ordersPrice')
			->setSortable()
			->setSortableCallback(function(array $data, array $sort) {
				$column    = key($sort);
				$direction = $sort[$column];

				usort($data, static function($a, $b) use ($column, $direction) {
					if ($a[$column] === $b[$column]) {
						return 0;
					}

					return ($a[$column] < $b[$column] ? -1 : 1) * ($direction === 'ASC' ? 1 : -1);
				});

				return $data;
			})
			->setRenderer(fn(array $row) => number_format($row['price'], 2, ',', ' '))
			->addCellAttributes(['class' => 'price border-right-stronger']);

		$grid->setColumnsSummary(['inOrdersCount', 'count', 'price'])->setRenderer(function($val, $col) {
			if ($col === 'price') {
				return number_format($val, 2, ',', ' ');
			}

			return (string) $val;
		});

		$grid->getColumn('inOrdersCount')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('count')->getElementPrototype('th')->addClass('w1nw');

		// Export
		$grid->addExportCsv('CSV Export', 'products.csv', 'utf-8');

		return $grid;
	}

	protected function getData(): array
	{
		if ($this->cData === null) {
			$this->cData                   = $this->productStatistics->getByProduct($this->from, $this->to, $this->site, $this->manufacturer, $this->supplier);
			$this->cData['featureColumns'] = [];
			$this->cData['featureValues']  = [];
			$this->cData['productsData']   = [];

			$featuresIds     = EshopStatisticsConfig::load('productStatisticGrid.featuresInGrid');
			$allowedFeatures = [];
			$allowedProducts = [];

			foreach ($this->features ? Json::decode($this->features, Json::FORCE_ARRAY) : [] as $featureId => $featureValue) {
				$allowedFeatures[] = $featureValue;
			}

			$productsIds = [];
			foreach ($this->cData['data'] as $data) {
				if (is_numeric($data['id'])) {
					$productsIds[] = $data['id'];
				}
			}

			if ($featuresIds && $productsIds) {
				foreach (array_chunk(array_unique($productsIds), 200) as $chunk) {
					foreach ($this->em->getConnection()->executeQuery("SELECT fp.id_product as product, fp.id_feature_value as valueId, fv.feature_id as featureId, ft.name as featureName, fvt.name as valueName
							FROM eshop_catalog__feature_product fp
							INNER JOIN eshop_catalog__feature_value fv ON fp.id_feature_value = fv.id AND fv.feature_id IN (" . implode(',', $featuresIds) . ")
							INNER JOIN eshop_catalog__feature_texts ft ON fv.feature_id = ft.id AND ft.lang = '" . $this->translator->getLocale() . "'
							INNER JOIN eshop_catalog__feature_value_texts fvt ON fv.id = fvt.id AND fvt.lang = '" . $this->translator->getLocale() . "'
							WHERE fp.id_product IN (" . implode(',', $chunk) . ")")->iterateAssociative() as $row) {
						$this->cData['productsData'][$row['product']]['features'][$row['featureId']] = $row['valueName'];
						$this->cData['featureColumns'][$row['featureId']]                            = $row['featureName'];
						$this->cData['featureValues'][$row['featureId']][$row['valueId']]            = $row['valueName'];

						if (in_array($row['valueId'], $allowedFeatures)) {
							$allowedProducts[] = $row['product'];
						}
					}
				}
			}

			foreach ($this->cData['featureValues'] as $k => &$v) {
				asort($v, SORT_NATURAL);
			}

			if ($allowedFeatures) {
				$this->cData['data'] = array_intersect_key($this->cData['data'], array_flip($allowedProducts));
			}

			$categories = [];
			foreach ($this->frontCategories->getAllRootIds() as $k => $v) {
				$categories[$k] = $this->frontCategories->getCategories($v, $this->translator->getLocale());
			}

			foreach (array_chunk(array_unique($productsIds), 400) as $chunk) {
				$tmp = [
					'pis.category_id = c.id',
					'pis.product_id IN (' . implode(',', $chunk) . ')',
				];

				if ($this->site) {
					$tmp[] = "pis.site = '" . $this->site . "'";
				}

				foreach ($this->em->getConnection()->executeQuery("SELECT pis.product_id, c.id, pis.site FROM eshop_catalog__category c
                     		INNER JOIN eshop_catalog__product_in_site pis ON " . implode(' AND ', $tmp))
					         ->iterateAssociative() as $row) {
					$cat                                                                         = $categories[$row['site']][$row['id']] ?? null;
					$this->cData['data'][(int) $row['product_id']]['mainCategory'][$row['site']] = $cat ? $cat->getParentPathStringFlipped() . ' > ' . $cat->name : '';
				}
			}

			uasort($this->cData['data'], static function($a, $b) {
				return $b['price'] <=> $a['price'];
			});
		}

		return $this->cData;
	}
}
