<?php declare(strict_types = 1);

namespace EshopStatistics\AdminModule\Model;

use Core\Model\Entities\QueryBuilder;
use Core\Model\Helpers\BaseEntityService;
use DateTimeInterface;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\Model\Config;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\Statistics\BaseStatistics;
use EshopStatistics\Model\EshopStatisticsConfig;
use EshopStock\Model\Repository\Supplies;
use Nette\DI\Container;
use Nette\Utils\DateTime;

/**
 * @method Order|null getReference($id)
 * @method Order[] getAll()
 * @method Order|null get($id)
 */
class ManufacturerStatistics extends BaseEntityService
{
	protected BaseStatistics $baseStatistics;
	protected Manufacturers  $manufacturers;
	protected Container      $container;

	public function __construct(
		BaseStatistics $baseStatistics,
		Manufacturers  $manufacturers,
		Container      $container
	)
	{
		$this->baseStatistics = $baseStatistics;
		$this->manufacturers  = $manufacturers;
		$this->container      = $container;
	}

	public function getByManufacturer(string $from, string $to): array
	{
		$withoutVat = EshopStatisticsConfig::load('priceWithoutVat', false);
		$fromDate   = $from ? new DateTime($from) : (new DateTime)->modify('-1 year');
		$toDate     = $to ? new DateTime($to) : (new DateTime);

		$fromDate->setTime(0, 0);
		$toDate->setTime(23, 59, 59);

		$manufacturersList = $this->manufacturers->getOptionsForSelect();

		$result = [];
		$query  = $this->baseStatistics->getBaseOrdersQB($fromDate, $toDate);
		$dates  = $this->baseStatistics->getCreated($fromDate, $toDate);

		$orders = [];
		foreach ($query->getQuery()->getScalarResult() as $row) {
			$row['items'] = [];

			$orders[$row['id']] = $row;
		}

		$cancelled = $this->baseStatistics->getCancelled(array_keys($orders));

		$qbCallback   = static function(QueryBuilder &$qb) {
			if (!in_array('p', $qb->getAllAliases(), true)) {
				$qb->leftJoin('oi.product', 'p');
			}

			$qb->addSelect('IDENTITY(p.manufacturer) as manufacturer');
		};
		$dataCallback = static function(array &$item, array &$row, array $items) {
			$item['manufacturer'] = $row['manufacturer'];
		};

		$stockedInByDate              = $this->baseStatistics->getStockedInDate($fromDate, $toDate);
		$productManufacturer          = [];
		$productPrices                = [];
		$stockedInDateByManufacturer  = [];
		$currentInStockByManufacturer = [];

		// Naskladneno
		foreach (array_chunk(array_keys($stockedInByDate), 200) as $chunk) {
			foreach ($this->em->createQueryBuilder()->select('p.id, IDENTITY(p.manufacturer) as manufacturer, p.price')
				         ->from('EshopCatalog\Model\Entities\Product', 'p')
				         ->where('p.id IN (' . implode(',', $chunk) . ')')
				         ->getQuery()->getArrayResult() as $row) {
				$productManufacturer[$row['id']] = $row['manufacturer'];
				$productPrices[$row['id']]       = (float) $row['price'];

				if (!array_key_exists($row['manufacturer'], $stockedInDateByManufacturer)) {
					$stockedInDateByManufacturer[$row['manufacturer']] = [
						'count'         => 0,
						'purchasePrice' => 0,
					];
				}

				$stockedInDateByManufacturer[$row['manufacturer']]['count']         += $stockedInByDate[$row['id']]['count'];
				$stockedInDateByManufacturer[$row['manufacturer']]['purchasePrice'] += $stockedInByDate[$row['id']]['purchasePrice'];
			}

			foreach ($this->em->createQueryBuilder()
				         ->select('COUNT(sp) as count, SUM(sp.purchasePrice) as purchasePrice, IDENTITY(sp.product) as product')
				         ->from('EshopStock\Model\Entities\SupplyProduct', 'sp')
				         ->where('sp.product IN (' . implode(',', $chunk) . ')')
				         ->andWhere('sp.order IS NULL')
				         ->andWhere('sp.writtenOffDate IS NULL')
				         ->groupBy('sp.product')
				         ->getQuery()->getArrayResult() as $row) {
				if (!array_key_exists($productManufacturer[$row['product']], $currentInStockByManufacturer)) {
					$currentInStockByManufacturer[$productManufacturer[$row['product']]] = [
						'count'         => 0,
						'purchasePrice' => 0,
						'value'         => 0,
					];
				}
				$currentInStockByManufacturer[$productManufacturer[$row['product']]]['count']         += $row['count'];
				$currentInStockByManufacturer[$productManufacturer[$row['product']]]['purchasePrice'] += $row['purchasePrice'];
				$currentInStockByManufacturer[$productManufacturer[$row['product']]]['value']         += ($row['count'] * $productPrices[$row['product']]);
			}
		}

		foreach ($this->baseStatistics->getOrderItems(array_keys($orders), $qbCallback, $dataCallback) as $orderId => $orderItems) {

			/** @var DateTimeInterface|null $created */
			$created = $dates[$orderId];
			if (!$created || isset($cancelled[$orderId])) {
				continue;
			}

			$orderAdded = false;
			foreach ($orderItems as $item) {
				$manu = $item['manufacturer'];

				if (!$manu) {
					continue;
				}

				if (!isset($result[$manu])) {
					$productsSold[$manu] = 0;
					$result[$manu]       = [
						'manufacturer'             => $manu,
						'name'                     => $manufacturersList[$manu],
						'orders'                   => 0,
						'purchasePrice'            => 0,
						'price'                    => 0,
						'priceWithoutVat'          => 0,
						'profit'                   => 0,
						'currentInStock'           => $currentInStockByManufacturer[$manu]['count'] ?? 0,
						'currentInStockValue'      => 0,
						'currentInStockValueRatio' => 0,
					];
				}

				if ($withoutVat) {
					$tmp = OrderHelper::getItemPriceTotalWithoutVat($item['price'], $item['quantity'], $item['vatRate'], []);

					$price           = $tmp;
					$priceWithoutVat = $tmp;
				} else {
					$price           = OrderHelper::getItemPriceTotal($item['price'], $item['quantity'], []);
					$priceWithoutVat = Config::load('enablePurchasePrice')
						? OrderHelper::getItemPriceTotalWithoutVat($item['price'], $item['quantity'], $item['vatRate'], [])
						: 0;
				}

				$costs = Config::load('enablePurchasePrice') ? $item['purchasePrice'] : 0;

				$result[$manu]['price']           += $price;
				$result[$manu]['priceWithoutVat'] += $priceWithoutVat;
				$result[$manu]['purchasePrice']   += $costs;
				$result[$manu]['profit']          += $priceWithoutVat - $costs;

				if (!$orderAdded) {
					$result[$manu]['orders']++;
				}

				$orderAdded = true;
			}
		}

		if (class_exists('EshopStock\Model\Repository\Supplies')) {
			foreach ($result as &$row) {
				/** @var Supplies $supplies */
				$supplies = $this->container->getService('eshopStock.admin.supplies');

				$tmp = 0;
				foreach ($this->em->getConnection()->fetchAllAssociative("SELECT id FROM eshop_stock__stock") as $stock) {
					$filter = $supplies->createQueryFilter((int) $stock['id']);
					$filter->toDate((new DateTime())->setTime(23, 59, 59));
					$filter->withManufacturer((int) $row['manufacturer']);

					$tmp += $filter->getSumSimplified()['itemsValue'] ?? 0;
				}

				$row['currentInStockValue'] = $tmp;
			}
		}


		foreach ($result as &$row) {
			$row['profitPercent']            = round(($row['profit'] / $row['priceWithoutVat']) * 100, 2);
			$row['currentInStockValueRatio'] = round(($row['currentInStockValue'] / $row['purchasePrice']) * 100, 2);
		}
		unset($row);

		usort($result, static fn($a, $b) => $a['name'] <=> $b['name']);

		return $result;
	}
}
