<?php declare(strict_types = 1);

namespace EshopStatistics\CronModule\Presenters;

use Core\Model\Entities\EntityManagerDecorator;
use Currency\Model\Currencies;
use EshopOrders\Model\EshopOrdersConfig;
use EshopStatistics\AdminModule\Model\OrderStatistics;
use EshopStatistics\AdminModule\Model\ProductStatistics;
use Nette\Security\Passwords;
use Nette\Utils\DateTime;

class StatisticsPresenter extends \Core\FrontModule\Presenters\BasePresenter
{
	protected OrderStatistics        $orderStatistics;
	protected ProductStatistics      $productStatistics;
	protected Currencies             $currencies;
	protected EntityManagerDecorator $emStats;

	public function __construct(
		OrderStatistics        $orderStatistics,
		ProductStatistics      $productStatistics,
		Currencies             $currencies,
		EntityManagerDecorator $emStats
	)
	{
		parent::__construct();
		$this->orderStatistics   = $orderStatistics;
		$this->productStatistics = $productStatistics;
		$this->currencies        = $currencies;
		$this->emStats           = $emStats;
	}

	public function actionDaily(string $key, string $siteIdent, ?string $currency = null): void
	{
		$password  = new Passwords;
		$keyVerify = $password->verify(EshopOrdersConfig::load('cronUrlHash') . $siteIdent, $key);

		if (!$keyVerify) {
			$this->error();
		}

		$fromModify = $this->getParameter('from', null) ?: '2 years';

		$to   = (new DateTime())->modify('-1 day')->setTime(23, 59, 59);
		$from = $to->modifyClone('-' . $fromModify)->setTime(0, 0, 0);

		header('Content-Encoding: UTF-8');
		header("content-type:application/csv;charset=UTF-8");
		header("Content-Disposition:attachment;filename=\"statistiky.csv\"");
		header('Content-Transfer-Encoding: binary');

		$fp = fopen('php://output', 'wb');

		if ($fp === false) {
			exit;
		}

		fputs($fp, "\xEF\xBB\xBF");

		fputcsv($fp, [
			'Date',
			'Orders count',
			'Profit',
			'Profit percent',
			'Total price',
			'Total price B2C',
			'Total price B2B',
		], ';');

		$toDate = $to->format('Y-m-d');
		foreach ($this->orderStatistics->getStatisticsByShop($from, $to, 'Y-m-d', $siteIdent, [], null, true) as $date => $row) {
			$totalPriceB2C = 0;
			$totalPriceB2B = 0;
			$totalPrice = 0;

			if ($currency) {
				$totalPrice += $row['ordersPrice-' . $siteIdent . '-' . $currency] ?? 0;
				$totalPriceB2C += $row['ordersPrice-' . $siteIdent . '-' . $currency . 'B2C'] ?? 0;
				$totalPriceB2B += $row['ordersPrice-' . $siteIdent . '-' . $currency . 'B2B'] ?? 0;
			} else {
				$totalPrice += $row['ordersPrice-' . $siteIdent ] ?? 0;
				$totalPriceB2C += $row['ordersPrice-' . $siteIdent . 'B2C'] ?? 0;
				$totalPriceB2B += $row['ordersPrice-' . $siteIdent . 'B2B'] ?? 0;
			}

			if ($date > $toDate) {
				continue;
			}

			$data = [
				$date,
				$row['orders-' . $siteIdent],
				number_format($row['ordersProfit-' . $siteIdent], 2, ',', ''),
				number_format(((float) $row['ordersProfitPercent-' . $siteIdent] / 100), 2, ',', ''),
				number_format($totalPrice, 2, ',', ''),
				number_format($totalPriceB2C, 2, ',', ''),
				number_format($totalPriceB2B, 2, ',', ''),
			];

			fputcsv($fp, $data, ';');
		}

		fclose($fp);

		exit;
	}

	public function actionProducts(string $key, string $siteIdent, string $from): void
	{
		$password  = new Passwords;
		$keyVerify = $password->verify(EshopOrdersConfig::load('cronUrlHash') . $siteIdent, $key);
		$from      = DateTime::createFromFormat('Y-m-d', $from);

		if (!$keyVerify || !$from) {
			$this->error();
		}

		header('Content-Encoding: UTF-8');
		header("content-type:application/csv;charset=UTF-8");
		header("Content-Disposition:attachment;filename=\"statistiky-produkty.csv\"");
		header('Content-Transfer-Encoding: binary');

		$fp = fopen('php://output', 'wb');

		if ($fp === false) {
			exit;
		}

		fputs($fp, "\xEF\xBB\xBF");

		$to = new DateTime();
		foreach ($this->productStatistics->getByProduct($from->format('Y-m-d'), $to->format('Y-m-d'), $siteIdent)['data'] as $row) {
			if ($row['price'] <= 0) {
				continue;
			}

			fputcsv($fp, [
				$row['id'],
				$row['count'],
				number_format((float) $row['price'], 2, '.', '') . ' ' . $this->currencies->getDefaultCode(),
			], ';');
		}

		fclose($fp);

		exit;
	}

	public function actionZboziReport(string $key, string $siteIdent): void
	{
		$password  = new Passwords;
		$keyVerify = $password->verify(EshopOrdersConfig::load('cronUrlHash') . $siteIdent, $key);
		$monthDate = (new DateTime())->modify('-1 month')->setTime(0, 0, 0);

		if (!$keyVerify) {
			$this->error();
		}

		$yearMonth = $monthDate->format('Y-m');

		$from = (clone $monthDate)->modify('first day of this month')->setTime(0, 0, 0);
		$to   = (clone $monthDate)->modify('last day of this month')->setTime(23, 59, 59);

		$reportId = $this->emStats->getConnection()->fetchOne('SELECT id 
				FROM eshop_statistics__zbozi_report 
				WHERE site_ident = :siteIdent AND `from` = :from AND `to` = :to
				LIMIT 1', [
			':siteIdent' => $siteIdent,
			':from'      => $from->format('Y-m-d H:i:s'),
			':to'        => $to->format('Y-m-d H:i:s'),
		]);

		if (!$reportId) {
			$this->error();
		}

		header('Content-Encoding: UTF-8');
		header("content-type:application/csv;charset=UTF-8");
		header("Content-Disposition:attachment;filename=\"statistiky-zbozi-report-{$yearMonth}.csv\"");
		header('Content-Transfer-Encoding: binary');

		$fp = fopen('php://output', 'wb');

		if ($fp === false) {
			exit;
		}

		fputs($fp, "\xEF\xBB\xBF");

		fputcsv($fp, [
			'itemId',
			'itemTitle',
			'views',
			'clicks',
			'cost',
			'conversions',
			'conversionsValue',
		], ';');
		foreach ($this->emStats->getConnection()->fetchAllAssociative("SELECT * 
    				FROM eshop_statistics__zbozi_item_report 
					WHERE zbozi_report_id = :reportId", [
			':reportId' => $reportId,
		]) as $row) {
			fputcsv($fp, [
				$row['item_id'],
				$row['item_name'],
				(string) $row['views'],
				(string) $row['clicks'],
				'"' . number_format((float) $row['costs'] / 100, 2, '.', '') . '"',
				(string) $row['conversions'],
				'"' . number_format((float) $row['conversions_value'] / 100, 2, '.', '') . '"',
			], ';');
		}

		fclose($fp);

		exit;
	}

	public function actionZboziAggregatedReport(string $key, string $siteIdent): void
	{
		$limit = $this->getParameter('limit') ?: 14;
		$password  = new Passwords;
		$keyVerify = $password->verify(EshopOrdersConfig::load('cronUrlHash') . $siteIdent, $key);

		if (!$keyVerify) {
			$this->error();
		}

		header('Content-Encoding: UTF-8');
		header("content-type:application/csv;charset=UTF-8");
		header("Content-Disposition:attachment;filename=\"statistiky-zbozi-aggregated-report.csv\"");
		header('Content-Transfer-Encoding: binary');

		$fp = fopen('php://output', 'wb');

		if ($fp === false) {
			exit;
		}

		fputs($fp, "\xEF\xBB\xBF");

		fputcsv($fp, [
			'date',
			'views',
			'clicks',
			'cost',
			'conversions',
			'conversionsValue',
		], ';');
		foreach ($this->emStats->getConnection()->fetchAllAssociative("SELECT * 
    				FROM eshop_statistics__zbozi_aggregated_report 
    				ORDER BY `day` DESC
    				LIMIT " . $limit) as $row) {
			fputcsv($fp, [
				$row['day'],
				(string) $row['views'],
				(string) $row['clicks'],
				number_format((float) $row['costs'] / 100, 2, '.', ''),
				(string) $row['conversions'],
				number_format((float) $row['conversions_value'] / 100, 2, '.', ''),
			], ';');
		}

		fclose($fp);

		exit;
	}
}
