<?php declare(strict_types = 1);

namespace EshopStatistics\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use Nette\Utils\DateTime;
use Nette\Utils\Json;
use Tracy\Debugger;

class ZboziAggregatedReportService
{
	protected EntityManagerDecorator $em;
	protected Sites                  $sites;

	public function __construct(
		EntityManagerDecorator $em,
		Sites                  $sites
	)
	{
		$this->em    = $em;
		$this->sites = $sites;
	}

	public function downloadLastDay(string $dayModifier = '-1 day', int $days = 1): void
	{
		$now = new DateTime();

		$from = (clone $now)->modify($dayModifier);
		$from = $from->setTime(12, 0, 0);

		$to = clone $from;
		if ($days > 1) {
			$to = $to->modify('+' . ($days - 1) . ' days');
		}

		foreach ($this->sites->getSites() as $site) {
			$this->downloadReport($site->getIdent(), $from, $to);
		}
	}

	public function downloadReport(string $siteIdent, DateTime $from, DateTime $to): void
	{
		$username = EshopStatisticsConfig::load('zboziApi.' . $siteIdent . '.username');
		$password = EshopStatisticsConfig::load('zboziApi.' . $siteIdent . '.password');

		if (!$username || !$password) {
			return;
		}

		$authorization = base64_encode("$username:$password");

		try {
			$curl = curl_init();

			$dateFrom = (clone $from)->setTime(0, 0, 0);
			$dateTo   = (clone $to)->setTime(23, 59, 59);

			curl_setopt_array($curl, [
				CURLOPT_URL            => 'https://api.zbozi.cz/v1/shop/statistics/aggregated?timestampFrom=' . $dateFrom->getTimestamp() . '&timestampTo=' . $dateTo->getTimestamp() . '&granularity=daily',
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_ENCODING       => '',
				CURLOPT_MAXREDIRS      => 10,
				CURLOPT_TIMEOUT        => 0,
				CURLOPT_FOLLOWLOCATION => true,
				CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
				CURLOPT_CUSTOMREQUEST  => 'GET',
				CURLOPT_HEADER         => false,
				CURLOPT_HTTPHEADER     => [
					'Authorization: Basic ' . $authorization,
				],
			]);

			$response = curl_exec($curl);
			$httpCode = (int) curl_getinfo($curl, CURLINFO_HTTP_CODE);
			curl_close($curl);

			if ($httpCode === 200) {
				$data = Json::decode((string) $response, Json::FORCE_ARRAY);

				foreach ($data['data'] ?? [] as $row) {
					$day    = DateTime::from($row['startTimestamp']);
					$dayStr = $day->format('Y-m-d');

					$exist = $this->em->getConnection()->fetchOne('SELECT COUNT(*) 
								FROM eshop_statistics__zbozi_aggregated_report 
								WHERE site_ident = :siteIdent AND `day` = :day
								LIMIT 1', [
						':siteIdent' => $siteIdent,
						':day'       => $dayStr,
					]);

					if ($exist) {
						continue;
					}

					$this->em->getConnection()->insert('eshop_statistics__zbozi_aggregated_report', [
						'site_ident'        => $siteIdent,
						'day'               => $dayStr,
						'views'             => (int) $this->getTotalValueFromFieldResponse($row['views']),
						'clicks'            => (int) $this->getTotalValueFromFieldResponse($row['clicks']),
						'costs'             => (int) round($this->getTotalValueFromFieldResponse($row['cost']) * 100, 0),
						'conversions'       => (int) $this->getTotalValueFromFieldResponse($row['conversions']),
						'conversions_value' => (int) round($this->getTotalValueFromFieldResponse($row['conversionsValue']) * 100, 0),
					]);
				}
			} else {
				Debugger::log('Zbozi API error: ' . $httpCode, 'zbozi-api');
				Debugger::log('Zbozi API error: ' . $response, 'zbozi-api');
			}
		} catch (\Exception $e) {
			Debugger::log($e, 'zbozi-api');
		}
	}

	protected function getTotalValueFromFieldResponse(array $row): float
	{
		return (float) ($row['categoryListing'] ?? 0)
			+ (float) ($row['categorySearch'] ?? 0)
			+ (float) ($row['productDetail'] ?? 0)
			+ (float) ($row['search'] ?? 0)
			+ (float) ($row['topProductDetail'] ?? 0);
	}
}
