<?php declare(strict_types = 1);

namespace EshopStatistics\AdminModule\Components\Manufacturer;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use DateTime;
use EshopStatistics\AdminModule\Model\ManufacturerStatistics;
use EshopStock\DI\EshopStockExtension;
use Nette\Application\Attributes\Persistent;
use Nette\Utils\ArrayHash;

class ManufacturerStatisticsGrid extends BaseControl
{
	#[Persistent]
	public string $from;

	#[Persistent]
	public string $to;

	public function __construct(
		protected ManufacturerStatistics $manufacturerStatistics,
	)
	{
		$this->from = (new DateTime())->modify('-1 year')->format('Y-m-d');
		$this->to   = (new DateTime())->format('Y-m-d');
	}

	public function render(): void
	{
		if (class_exists(EshopStockExtension::class)) {
			$this->template->render($this->getTemplateFile());
		} else {
			$this->template->render($this->getTemplateFile('stockNotFound.latte'));
		}
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentFilterForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addDatePicker('from', '')
			->setHtmlAttribute('data-type', 'yearMonthDay')
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();') //TODO s naja se nacita nejaky jiny datepicker
			->setDefaultValue($this->from);
		$form->addDatePicker('to', '')
			->setHtmlAttribute('data-type', 'yearMonthDay')
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();')
			->setDefaultValue($this->to);

		$form->addSubmit('update', 'default.update');

		$form->onSuccess[] = function(BaseForm $form, ArrayHash $values): void {
			$this->from = $values->from->format('Y-m-d');
			$this->to   = $values->to->format('Y-m-d');

			$this['grid']->reload();
			$this->redrawControl('card');
		};

		return $form;
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$grid->setItemsPerPageList([], true);
		$grid->setDefaultSort('name');

		$qb = $this->manufacturerStatistics->getByManufacturer($this->from, $this->to);

		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnText('name', 'eshopStatistics.manufacturerStatistics.name')
			->setSortable();
		$grid->addColumnNumber('orders', 'eshopStatistics.manufacturerStatistics.orders')
			->setFormat(0, '', ' ')
			->setSortable();
		$grid->addColumnNumber('purchasePrice', 'eshopStatistics.manufacturerStatistics.purchasePrice')
			->setFormat(2, ',', ' ')
			->setSortable();
		$grid->addColumnNumber('price', 'eshopStatistics.manufacturerStatistics.price')
			->setFormat(2, ',', ' ')
			->setSortable();
		$grid->addColumnNumber('profit', 'eshopStatistics.manufacturerStatistics.profit')
			->setFormat(2, ',', ' ')
			->setSortable();
		$grid->addColumnNumber('profitPercent', 'eshopStatistics.manufacturerStatistics.profitPercent')
			->setFormat(2, ',', ' ')
			->setSortable();
		$grid->addColumnNumber('currentInStock', 'eshopStatistics.manufacturerStatistics.currentInStock')
			->setFormat(0, '', ' ')
			->setSortable();
		$grid->addColumnNumber('currentInStockValue', 'eshopStatistics.manufacturerStatistics.currentInStockValue')
			->setFormat(2, ',', ' ')
			->setSortable();
		$grid->addColumnNumber('currentInStockValueRatio', 'eshopStatistics.manufacturerStatistics.currentInStockValueRatio')
			->setFormat(2, ',', ' ')
			->setSortable();

		$grid->setColumnsSummary(['orders', 'purchasePrice', 'price', 'profit', 'currentInStock', 'currentInStockValue']);

		return $grid;
	}
}
