<?php declare(strict_types = 1);

namespace EshopStatistics\AdminModule\Components\Orders;

use Core\Model\Dao\Site;
use Core\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use DateTime;
use EshopCatalog\Model\Config;
use EshopStatistics\AdminModule\Model\OrderStatistics;
use EshopStatistics\Model\EshopStatisticsConfig;
use Nette\Application\Attributes\Persistent;
use Nette\Utils\ArrayHash;
use Ublaboo\DataGrid\Column\ColumnNumber;
use Ublaboo\DataGrid\DataSource\ArrayDataSource;

class OrderStatisticsGrid extends BaseControl
{
	#[Persistent]
	public string $from;

	#[Persistent]
	public string $to;

	#[Persistent]
	public string $showHidden = '0';

	public function __construct(
		protected OrderStatistics $orderStatistics,
		protected Sites           $sites,
	)
	{
		$this->from = (new DateTime)->modify('-1 year')->format('Y-m-d');
		$this->to   = (new DateTime)->format('Y-m-d');
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentFromToForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addDatePicker('from', '')
			->setHtmlAttribute('data-type', 'yearMonth')
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();') //TODO po naja reloadu nefunguje itemDetail
			->setDefaultValue($this->from);
		$form->addDatePicker('to', '')
			->setHtmlAttribute('data-type', 'yearMonth')
			->setHtmlAttribute('onChange', '$(this.form).closest(\'form\').first().submit();') //po naja reloadu nefunguje itemDetail
			->setDefaultValue($this->to);
		$form->addCheckbox('showHidden', 'eshopStatistics.orderStatistics.showHidden')
			->setDefaultValue($this->showHidden);

		$form->addSubmit('update', 'default.update');

		$form->onSuccess[] = function(BaseForm $form, ArrayHash $values): void {
			$this->from       = $values->from->format('Y-m-d');
			$this->to         = $values->to->format('Y-m-d');
			$this->showHidden = (string) $values->showHidden;

			$this->redrawControl('ordersGrid');
		};

		return $form;
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setTemplateFile(__DIR__ . '/../StatisticsGridHeader_grid.latte');
		$grid->setItemsPerPageList([]);

		$from = new DateTime($this->from);
		$to   = new DateTime($this->to);

		/** @var Site[] $sites */
		$sites = [];

		if ($this->showHidden === '0') {
			foreach ($this->sites->getSites(false) as $site) {
				if ($site->isVisible) {
					$sites[$site->getIdent()] = $site;
				}
			}
		} else {
			$sites = $this->sites->getSites(false);
		}

		$arrayData = $this->orderStatistics->getMonthlyByShop($from, $to, array_keys($sites));

		$dataSource = new ArrayDataSource($arrayData);
		$dataSource->getData();
		$grid->setDataSource($dataSource);

		$exportColumns = [];
		$summaryCols   = [];

		// Columns
		$grid->addColumnDateTime('date', 'eshopStatistics.orderStatistics.monthlyOrders.date')
			->setFormat('m/Y')
			->addCellAttributes(['class' => 'border-right-stronger'])
			->setTemplate(null, ['supHeaderGroup' => 'date']);
		$exportColumns['date'] = $grid->getColumn('date');

		$grid->addColumnNumber('orders', 'eshopStatistics.orderStatistics.monthlyOrders.ordersCount')
			->setFormat(0, '', ' ')
			->setTemplate(null, [
				'supHeaderText'  => $this->translator->translate('eshopStatistics.orderStatistics.monthlyOrders.total'),
				'supHeaderGroup' => 'total',
			]);
		$exportColumns['orders'] = $grid->getColumn('orders');
		$summaryCols[]           = 'orders';

		if (Config::load('enablePurchasePrice')) {
			$grid->addColumnNumber('ordersProfit', 'eshopStatistics.orderStatistics.monthlyOrders.profit')
				->setFormat(2, ',', ' ')
				->addCellAttributes(['class' => 'price'])
				->setTemplate(null, ['supHeaderGroup' => 'total']);
			$exportColumns['ordersProfit'] = $grid->getColumn('ordersProfit');
			$summaryCols[]                 = 'ordersProfitPercent';

			$grid->addColumnNumber('ordersProfitPercent', 'eshopStatistics.orderStatistics.monthlyOrders.profitPercent')
				->setFormat(2, ',', ' ')
				->addCellAttributes(['class' => 'price'])
				->setTemplate(null, ['supHeaderGroup' => 'total']);
			$exportColumns['ordersProfitPercent'] = $grid->getColumn('ordersProfitPercent');
			$summaryCols[]                        = 'ordersProfit';
		}

		$moreSumColumns = (array) EshopStatisticsConfig::load('ordersStatisticGrid.sumCurrencies') ?: [];

		$grid->addColumnNumber('ordersPrice', 'eshopStatistics.orderStatistics.monthlyOrders.ordersPrice')
			->setFormat(2, ',', ' ')->addCellAttributes(['class' => 'price'])
			->addCellAttributes(['class' => 'price'])
			->setTemplate(null, ['supHeaderGroup' => 'total']);
		$exportColumns['ordersPrice'] = $grid->getColumn('ordersPrice');
		$summaryCols[]                = 'ordersPrice';

		$grid->addColumnNumber('ordersPriceCZK', 'eshopStatistics.orderStatistics.monthlyOrders.priceCZK')
			->setFormat(2, ',', ' ')
			->addCellAttributes(['class' => 'price'])
			->setTemplate(null, ['supHeaderGroup' => 'total']);
		$exportColumns['ordersPriceCZK'] = $grid->getColumn('ordersPriceCZK');
		$summaryCols[]                   = 'ordersPriceCZK';

		$grid->addColumnNumber('ordersPriceEUR', 'eshopStatistics.orderStatistics.monthlyOrders.priceEUR')
			->setFormat(2, ',', ' ')
			->addCellAttributes(['class' => 'price' . (empty($moreSumColumns) ? ' border-right-stronger' : '')])
			->setTemplate(null, ['supHeaderGroup' => 'total']);
		$exportColumns['ordersPriceEUR'] = $grid->getColumn('ordersPriceEUR');
		$summaryCols[]                   = 'ordersPriceEUR';

		$moreSumColumnsCount = count($moreSumColumns);
		foreach ($moreSumColumns as $k => $curr) {
			$grid->addColumnNumber('ordersPrice' . $curr, $this->t('eshopStatistics.orderStatistics.monthlyOrders.priceX', ['curr' => $curr]))
				->setFormat(2, ',', ' ')
				->setTranslatableHeader(false)
				->addCellAttributes(['class' => 'price' . ($k === $moreSumColumnsCount - 1 ? ' border-right-stronger' : '')])
				->setTemplate(null, ['supHeaderGroup' => 'total']);
			$exportColumns['ordersPrice' . $curr] = $grid->getColumn('ordersPrice' . $curr);
			$summaryCols[]                        = 'ordersPrice' . $curr;
		}

		krsort($sites);
		foreach ($sites as $site) {
			$domain    = $site->getDomains()[$this->translator->getLocale()] ?? $site->getDefaultDomain();

			if (!$domain) {
				continue;
			}

			$siteName  = $domain->siteName;
			$siteIdent = $site->getIdent();

			$moreSiteColumns = (array) EshopStatisticsConfig::load('ordersStatisticGrid.shopCurrencies.' . $siteIdent) ?: [];

			$grid->addColumnNumber('orders-' . $siteIdent, 'eshopStatistics.orderStatistics.monthlyOrders.ordersCount')
				->setFormat(0, '', ' ')
				->setTemplate(null, [
					'supHeaderText'  => $siteName . ' (' . $siteIdent . ')',
					'supHeaderGroup' => $siteIdent,
				]);
			$exportColumns['orders-' . $siteIdent] = (new ColumnNumber($grid, 'orders-' . $siteIdent, 'orders-' . $siteIdent, $siteIdent . ' - ' . $this->t('eshopStatistics.orderStatistics.monthlyOrders.ordersCount')))
				->setFormat(0, '', ' ')
				->setTranslatableHeader(false);
			$summaryCols[]                         = 'orders-' . $siteIdent;

			if (Config::load('enablePurchasePrice')) {
				$grid->addColumnNumber('ordersProfit-' . $siteIdent, 'eshopStatistics.orderStatistics.monthlyOrders.profit')
					->setFormat(2, ',', ' ')
					->addCellAttributes(['class' => 'price'])
					->setTemplate(null, ['supHeaderGroup' => $siteIdent]);
				$exportColumns['ordersProfit-' . $siteIdent] = (new ColumnNumber($grid, 'ordersProfit-' . $siteIdent, 'ordersProfit-' . $siteIdent, $siteIdent . ' - ' . $this->t('eshopStatistics.orderStatistics.monthlyOrders.profit')))
					->setFormat(2, ',', ' ')
					->setTranslatableHeader(false);
				$summaryCols[]                               = 'ordersProfitPercent-' . $siteIdent;

				$grid->addColumnNumber('ordersProfitPercent-' . $siteIdent, 'eshopStatistics.orderStatistics.monthlyOrders.profitPercent')
					->setFormat(2, ',', ' ')
					->addCellAttributes(['class' => 'price'])
					->setTemplate(null, ['supHeaderGroup' => $siteIdent]);
				$exportColumns['ordersProfitPercent-' . $siteIdent] = (new ColumnNumber($grid, 'ordersProfitPercent-' . $siteIdent, 'ordersProfitPercent-' . $siteIdent, $siteIdent . ' - ' . $this->t('eshopStatistics.orderStatistics.monthlyOrders.profitPercent')))
					->setFormat(2, ',', ' ')
					->setTranslatableHeader(false);
				$summaryCols[]                                      = 'ordersProfit-' . $siteIdent;
			}

			$grid->addColumnNumber('ordersPrice-' . $siteIdent, 'eshopStatistics.orderStatistics.monthlyOrders.ordersPrice')
				->setFormat(2, ',', ' ')
				->addCellAttributes(['class' => 'price'])
				->setTemplate(null, ['supHeaderGroup' => $siteIdent]);
			$exportColumns['ordersPrice-' . $siteIdent] = (new ColumnNumber($grid, 'ordersPrice-' . $siteIdent, 'ordersPrice-' . $siteIdent, $siteIdent . ' - ' . $this->t('eshopStatistics.orderStatistics.monthlyOrders.ordersPrice')))
				->setFormat(2, ',', ' ')
				->setTranslatableHeader(false);
			$summaryCols[]                              = 'ordersPrice-' . $siteIdent;

			$grid->addColumnNumber('ordersPrice-' . $siteIdent . '-CZK', 'eshopStatistics.orderStatistics.monthlyOrders.priceCZK')
				->setFormat(2, ',', ' ')
				->addCellAttributes(['class' => 'price'])
				->setTemplate(null, ['supHeaderGroup' => $siteIdent]);
			$exportColumns['ordersPrice-' . $siteIdent . '-CZK'] = (new ColumnNumber($grid, 'ordersPrice-' . $siteIdent . '-CZK', 'ordersPrice-' . $siteIdent . '-CZK', $siteIdent . ' - ' . $this->t('eshopStatistics.orderStatistics.monthlyOrders.priceCZK')))
				->setFormat(2, ',', ' ')
				->setTranslatableHeader(false);
			$summaryCols[]                                       = 'ordersPrice-' . $siteIdent . '-CZK';

			$grid->addColumnNumber('ordersPrice-' . $siteIdent . '-EUR', 'eshopStatistics.orderStatistics.monthlyOrders.priceEUR')
				->setFormat(2, ',', ' ')
				->addCellAttributes(['class' => 'price' . (empty($moreSiteColumns) ? ' border-right-stronger' : '')])
				->setTemplate(null, ['supHeaderGroup' => $siteIdent]);
			$exportColumns['ordersPrice-' . $siteIdent . '-EUR'] = (new ColumnNumber($grid, 'ordersPrice-' . $siteIdent . '-EUR', 'ordersPrice-' . $siteIdent . '-EUR', $siteIdent . ' - ' . $this->t('eshopStatistics.orderStatistics.monthlyOrders.priceEUR')))
				->setFormat(2, ',', ' ')
				->setTranslatableHeader(false);
			$summaryCols[]                                       = 'ordersPrice-' . $siteIdent . '-EUR';

			$moreSiteColumnsCount = count($moreSiteColumns);
			foreach ($moreSiteColumns as $k => $curr) {
				$grid->addColumnNumber('ordersPrice-' . $siteIdent . '-' . $curr, $this->t('eshopStatistics.orderStatistics.monthlyOrders.priceX', ['curr' => $curr]))
					->setFormat(2, ',', ' ')
					->setTranslatableHeader(false)
					->addCellAttributes(['class' => 'price' . ($k === $moreSiteColumnsCount - 1 ? ' border-right-stronger' : '')])
					->setTemplate(null, ['supHeaderGroup' => $siteIdent]);
				$exportColumns['ordersPrice-' . $siteIdent . '-' . $curr] = (new ColumnNumber($grid, 'ordersPrice-' . $siteIdent . '-' . $curr, 'ordersPrice-' . $siteIdent . '-' . $curr, $siteIdent . ' - ' . $this->t('eshopStatistics.orderStatistics.monthlyOrders.priceX', ['curr' => $curr])))
					->setFormat(2, ',', ' ')
					->setTranslatableHeader(false);
				$summaryCols[]                                            = 'ordersPrice-' . $siteIdent . '-' . $curr;
			}
		}

		$grid->setColumnsSummary($summaryCols)
			->setRenderer(function($v, $column) use (&$arrayData) {
				if (\str_contains($column, 'ordersProfitPercent')) {
					$tmp    = explode('-', $column);
					$price  = 0;
					$profit = 0;

					foreach ($arrayData as $vals) {
						if (isset($tmp[1])) {
							$price  += $vals['ordersPrice-' . $tmp[1]];
							$profit += $vals['ordersProfit-' . $tmp[1]];
						} else {
							$price  += $vals['ordersPrice'];
							$profit += $vals['ordersProfit'];
						}
					}

					return number_format(round(($profit / $price) * 100, 2), 2, ',', ' ');
				}

				if ($column === 'orders' || \str_starts_with($column, 'orders-')) {
					return number_format($v, 0, '', ' ');
				}

				return number_format($v, 2, ',', ' ');
			});
		$grid->setItemsDetail($this->getTemplateFile('OrderStatisticsGridDetail', 'Orders'));

		set_time_limit(300);
		$grid->addExportCsv('CSV Export', 'orders-statistics.csv', 'utf-8', ';', false, false)
			->setColumns($exportColumns);

		return $grid;
	}
}
