<?php declare(strict_types = 1);

namespace EshopStatistics\AdminModule\Model;

use Contributte\Translation\Translator;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\Parameters;
use Core\Model\Sites;
use DateInterval;
use DateTimeInterface;
use EshopCatalog\Model\Config;
use EshopOrders\AdminModule\Model\Payments;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Model\Entities\GroupCustomers;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\PriceCalculator\PriceCalculatorItem;
use EshopOrders\Model\Statistics\BaseStatistics;
use EshopStatistics\Model\EshopStatisticsConfig;
use Nette\Utils\DateTime;

class OrderStatistics extends BaseEntityService
{
	public static array $byShopHelper = [];

	protected array  $cLoaded   = [];
	protected ?array $cPayments = null;

	public function __construct(
		protected Translator     $translator,
		protected BaseStatistics $baseStatistics,
		protected Payments       $payments,
		protected Speditions     $speditions,
		protected Sites          $sites,
	)
	{
	}

	// -------------------- statistiky objednavek - pocet a cena ----------------------------------------------

	public function getDailyByShop(DateTimeInterface $from, DateTimeInterface $to): array
	{
		return $this->getStatisticsByShop($from, $to, 'Y-m-d');
	}

	public function getMonthlyByShop(?DateTimeInterface $from = null, ?DateTimeInterface $to = null, array $sites = []): array
	{
		return $this->getStatisticsByShop($from, $to, 'Y-m', null, $sites);
	}

	/**
	 * Statistiky objednavek - cena jen za zbozi bez dopravy
	 */
	public function getStatisticsByShop(
		?DateTimeInterface $from = null,
		?DateTimeInterface $to = null,
		string             $keyFormat = 'Y-m',
		?string            $siteIdent = null,
		array              $sites = [],
		?string            $groupBy = null,
		bool               $splitByCustomerGroupType = false
	): array
	{
		$groupBy = $groupBy ?: 'currency';

		$withoutVat = EshopStatisticsConfig::load('priceWithoutVat', false);
		$from       = $from instanceof DateTimeInterface ? new DateTime($from->format('Y-m-d')) : (new DateTime)->modify('-1 year');
		$to         = $to instanceof DateTimeInterface ? new DateTime($to->format('Y-m-d')) : (new DateTime);

		$from->modify('first day of this month')->setTime(0, 0, 0);
		$to->modify('last day of this month')->setTime(23, 59, 59);

		$result      = [];
		$dateCurrent = clone $from;
		while ($dateCurrent <= $to) {
			$key = $dateCurrent->format($keyFormat);
			if (!isset($result[$key])) {
				$result[$key] = $this->monthlyShopCreateRow(clone $dateCurrent, $keyFormat, $groupBy, $splitByCustomerGroupType);
			}

			if ($keyFormat === 'Y-m-d') {
				$dateCurrent->modify('+1 day');
			} else {
				$dateCurrent->modify('+1 month');
			}
		}

		$query = $this->baseStatistics->getBaseOrdersQB($from, $to, $siteIdent, $sites);
		$dates = $this->baseStatistics->getCreated($from, $to);

		$orders              = [];
		$customerGroupsTypes = [];

		if ($groupBy === 'country' || $splitByCustomerGroupType) {
			$query->leftJoin('o.addressInvoice', 'aInv')
				->leftJoin('o.addressDelivery', 'aDel');

			if ($groupBy === 'country') {
				$query->addSelect('IDENTITY(aInv.country) as countryInv, IDENTITY(aDel.country) as countryDel');
			}

			if ($splitByCustomerGroupType) {
				foreach ($this->em->getConnection()->executeQuery("SELECT id, `type` FROM eshop_orders__group_customers")->fetchAllAssociative() as $row) {
					$customerGroupsTypes[$row['id']] = $row['type'];
				}

				$query->addSelect('aInv.idNumber as idNumberInv');
				$query->leftJoin('o.customer', 'c')
					->addSelect('IDENTITY(c.groupCustomers) as groupCustomersId');
			}
		}

		foreach ($query->getQuery()->getScalarResult() as $row) {
			if ($groupBy === 'country') {
				$row['country'] = Strings::upper($row['countryDel'] ?: $row['countryInv'] ?: 'CZ');
			}

			$row['items']     = [];
			$row['discounts'] = [];

			$orders[$row['id']] = $row;
		}

		// Sleva objednavky
		foreach ($this->baseStatistics->getSales(array_keys($orders)) as $id => $sales) {
			$orders[$id]['discounts'] = $sales;
		}

		// Produkty
		foreach ($this->baseStatistics->getOrderItems(array_keys($orders)) as $id => $orderItems) {
			$orders[$id]['items'] = $orderItems;
		}

		// Invoice addr
		foreach ($this->baseStatistics->getAddrInvoice(array_keys($orders)) as $id => $addr) {
			$orders[$id]['addrInvoice'] = $addr;
		}

		// Zrusene
		$cancelled = $this->baseStatistics->getCancelled(array_keys($orders));

		foreach ($orders as $order) {
			/** @var DateTimeInterface|null $created */
			$created = $dates[$order['id']];
			if (!$created || isset($cancelled[$order['id']])) {
				continue;
			}

			$key = $created->format($keyFormat);

			$customerGroupType = null;

			if ($splitByCustomerGroupType) {
				if (isset($order['groupCustomersId'], $customerGroupsTypes[$order['groupCustomersId']])) {
					$customerGroupType = $customerGroupsTypes[$order['groupCustomersId']];
				} else if ($order['idNumberInv']) {
					$customerGroupType = GroupCustomers::typeB2B;
				} else {
					$customerGroupType = GroupCustomers::typeB2C;
				}
			}

			$price           = 0;
			$priceWithoutVat = 0;
			$costs           = 0;
			foreach ($order['items'] as $item) {
				$countryVatRate = null;
				if ($order['addrInvoice']) {
					$countryVatRate = OrderHelper::checkCountryVatRate(
						$item['vatRate'],
						$order['addrInvoice']['country'],
						(bool) $order['addrInvoice']['validatedVatNumber'],
						$order['addrInvoice']['idNumber'],
						$order['addrInvoice']['vatNumber'],
					);
				}

				$saleCalculatorItem = null;
				if ($item['saleValue']) {
					$itemSale           = OrderHelper::calculateSaleValue($item['saleType'], $item['saleValue'], $item['quantity'], $item['price']);
					$saleCalculatorItem = new PriceCalculatorItem('', (float) $itemSale, 1, $item['vatRate'], 'CZK');
				}

				$priceCalculatorItem = new PriceCalculatorItem('', $item['price'], $item['quantity'], $item['vatRate'], 'CZK');

				if ($withoutVat || $countryVatRate === 0) {
					$tmp = $saleCalculatorItem
						? $priceCalculatorItem->getTotalWithoutVat()->minus($priceCalculatorItem->getTotalWithoutVat())->getAmount()->toFloat()
						: $priceCalculatorItem->getTotalWithoutVat()->getAmount()->toFloat();

					$price           += $tmp;
					$priceWithoutVat += $tmp;
				} else {
					$price += $saleCalculatorItem
						? $priceCalculatorItem->getTotalWithVat()->minus($priceCalculatorItem->getTotalWithVat())->getAmount()->toFloat()
						: $priceCalculatorItem->getTotalWithVat()->getAmount()->toFloat();

					if (Config::load('enablePurchasePrice')) {
						$priceWithoutVat += $saleCalculatorItem
							? $priceCalculatorItem->getTotalWithoutVat()->minus($priceCalculatorItem->getTotalWithoutVat())->getAmount()->toFloat()
							: $priceCalculatorItem->getTotalWithoutVat()->getAmount()->toFloat();
					}
				}

				if (Config::load('enablePurchasePrice')) {
					$costs += $item['purchasePrice'] * $item['quantity'];
				}
			}

			foreach ($order['discounts'] as $discount) {
				$price += $discount;
			}

			$groupKey = $order['currencyCode'] ?: 'CZK';
			if ($groupBy === 'country') {
				$groupKey = $order['country'] ?: 'CZ';

				self::$byShopHelper['groupByCountries']['all'][$groupKey]               = $groupKey;
				self::$byShopHelper['groupByCountries'][$order['siteIdent']][$groupKey] = $groupKey;
			}

			$result[$key]['orders']++;
			$result[$key]['ordersPrice']             += $price;
			$result[$key]['ordersPrice' . $groupKey] += $price;

			if ($customerGroupType) {
				$result[$key]['orders' . $customerGroupType]++;
				$result[$key]['ordersPrice' . $customerGroupType] += $price;
				$result[$key]['orders' . $groupKey . $customerGroupType]++;
				$result[$key]['ordersPrice' . $groupKey . $customerGroupType] += $price;
			}

			$shop = $order['siteIdent'];
			$result[$key]['orders-' . $shop]++;
			$result[$key]['ordersPrice-' . $shop]                   += $price;
			$result[$key]['ordersPrice-' . $shop . '-' . $groupKey] += $price;

			if ($customerGroupType) {
				$result[$key]['orders-' . $shop . $customerGroupType]++;
				$result[$key]['ordersPrice-' . $shop . $customerGroupType] += $price;
				$result[$key]['orders-' . $shop . '-' . $groupKey . $customerGroupType]++;
				$result[$key]['ordersPrice-' . $shop . '-' . $groupKey . $customerGroupType] += $price;
			}

			if (Config::load('enablePurchasePrice')) {
				$result[$key]['ordersProfit']          += $priceWithoutVat - $costs;
				$result[$key]['ordersProfit-' . $shop] += $priceWithoutVat - $costs;
			}

			$orders[$order['id']] = null;
		}

		if (Config::load('enablePurchasePrice')) {
			foreach ($result as $k => $v) {
				if ($v['ordersProfit']) {
					$result[$k]['ordersProfitPercent'] = round(($v['ordersProfit'] / $v['ordersPrice']) * 100, 2);
				}

				foreach ($this->sites->getSites(false) as $site) {
					$ident = $site->getIdent();

					if ($v['ordersProfit-' . $ident]) {
						$result[$k]['ordersProfitPercent-' . $ident] = round(($v['ordersProfit-' . $ident] / $v['ordersPrice-' . $ident]) * 100, 2);
					}
				}
			}
		}

		krsort($result, SORT_STRING);

		return $result;
	}

	protected function monthlyShopCreateRow(\DateTime $dateTime, string $keyFormat, string $groupBy = 'currency', bool $splitByCustomerGroupType = false): array
	{
		$row = [
			'id'                  => $dateTime->format($keyFormat),
			'date'                => $dateTime,
			'orders'              => 0,
			'ordersPrice'         => 0,
			'ordersProfit'        => 0,
			'ordersProfitPercent' => 0,
		];

		if ($splitByCustomerGroupType) {
			foreach (GroupCustomers::getTypes() as $k => $v) {
				$row['orders' . $k]      = 0;
				$row['ordersPrice' . $k] = 0;
			}
		}

		if ($groupBy === 'currency') {
			foreach (Parameters::loadArray('currency.whitelist') ?: [] as $curr) {
				$row['ordersPrice' . $curr] = 0;

				if ($splitByCustomerGroupType) {
					foreach (GroupCustomers::getTypes() as $k => $v) {
						$row['orders' . $curr . $k]      = 0;
						$row['ordersPrice' . $curr . $k] = 0;
					}
				}
			}
		}

		foreach ($this->sites->getSites(false) as $site) {
			$row['orders-' . $site->getIdent()]              = 0;
			$row['ordersPrice-' . $site->getIdent()]         = 0;
			$row['ordersProfit-' . $site->getIdent()]        = 0;
			$row['ordersProfitPercent-' . $site->getIdent()] = 0;

			if ($splitByCustomerGroupType) {
				foreach (GroupCustomers::getTypes() as $k => $v) {
					$row['orders-' . $site->getIdent() . $k]      = 0;
					$row['ordersPrice-' . $site->getIdent() . $k] = 0;
				}
			}

			if ($groupBy === 'currency') {
				foreach (Parameters::loadArray('currency.whitelist') ?: [] as $curr) {
					$row['ordersPrice-' . $site->getIdent() . '-' . $curr] = 0;

					if ($splitByCustomerGroupType) {
						foreach (GroupCustomers::getTypes() as $k => $v) {
							$row['orders-' . $site->getIdent() . '-' . $curr . $k]      = 0;
							$row['ordersPrice-' . $site->getIdent() . '-' . $curr . $k] = 0;
						}
					}
				}
			}
		}

		return $row;
	}

	// -------------------- statistiky expedice - pro jednotlive dopravce ----------------------------------------------

	/**
	 * Statistiky dopravy - pocitame s cenou pouze za dopravu bez produktu
	 */
	public function getMonthlyByCarrier(?DateTimeInterface $from = null, ?DateTimeInterface $to = null): array
	{
		$withoutVat = EshopStatisticsConfig::load('priceWithoutVat', false);
		$from       = $from instanceof DateTimeInterface ? new DateTime($from->format('Y-m-d')) : (new DateTime)->modify('-1 year');
		$to         = $to instanceof DateTimeInterface ? new DateTime($to->format('Y-m-d')) : (new DateTime);

		$from->modify('first day of this month')->setTime(0, 0);
		$to->modify('last day of this month')->setTime(23, 59, 59);

		$baseKey = $from->format('Y-m') . '_' . $to->format('Y-m');
		if (isset($this->cLoaded[$baseKey])) {
			return $this->cLoaded[$baseKey];
		}

		$result      = [];
		$dateCurrent = clone $from;
		while ($dateCurrent <= $to) {
			$key = $dateCurrent->format('Y-m');
			if (!isset($result[$key])) {
				$result[$key] = $this->bySpeditionCreateRow(clone $dateCurrent);
			}
			$dateCurrent->add(new DateInterval('P1M'));
		}

		$orders = [];
		foreach ($this->baseStatistics->getBaseOrdersQB($from, $to)
			         ->addSelect('IDENTITY(p.payment) as paymentId, IDENTITY(s.spedition) as speditionId')
			         ->getQuery()->getScalarResult() as $row) {
			$orders[$row['id']] = $row;
		}

		$dates = $this->baseStatistics->getCreated($from, $to);

		foreach ($orders as $row) {
			$created = $dates[$row['id']] ?? null;

			if (!$created) {
				continue;
			}

			$paymentId    = $row['paymentId'];
			$paymentPrice = (float) $row['paymentPrice'];

			$speditionId    = $row['speditionId'];
			$speditionPrice = (float) $row['speditionPrice'];

			if ($withoutVat) {
				$paymentPrice   = OrderHelper::removeVat($paymentPrice, 21);
				$speditionPrice = OrderHelper::removeVat($speditionPrice, 21);
			}

			if ($this->getPaymentIdent($paymentId) === 'cod') {
				$speditionPrice += $paymentPrice;
			}

			$key = $created->format('Y-m');

			$result[$key]['orders-count-total']++;
			$result[$key]['orders-price-total'] += $paymentPrice + $speditionPrice;

			if ($paymentId) {
				$result[$key]['payment-count-total']++;
				$result[$key]['payment-price-total'] += $paymentPrice;
				$result[$key]['payment-count-' . $paymentId]++;
				$result[$key]['payment-price-' . $paymentId] += $paymentPrice;
			}

			if ($speditionId) {
				$result[$key]['spedition-count-total']++;
				$result[$key]['spedition-price-total'] += $speditionPrice;
				$result[$key]['spedition-count-' . $speditionId]++;
				$result[$key]['spedition-price-' . $speditionId] += $speditionPrice;
			}
		}

		krsort($result, SORT_STRING);
		$this->cLoaded[$baseKey] = $result;

		return $result;
	}

	protected function bySpeditionCreateRow(\DateTime $dateTime): array
	{
		$row = [
			'date'                  => $dateTime,
			'orders-count-total'    => 0, // jen expedovane objednavky
			'orders-price-total'    => 0,
			'payment-count-total'   => 0,
			'payment-price-total'   => 0,
			'spedition-count-total' => 0,
			'spedition-price-total' => 0,
		];
		foreach (array_keys($this->payments->getForSelectOption()) as $k) {
			$row['payment-count-' . $k] = 0;
			$row['payment-price-' . $k] = 0;
		}
		foreach (array_keys($this->speditions->getForSelectOption()) as $k) {
			$row['spedition-count-' . $k] = 0;
			$row['spedition-price-' . $k] = 0;
		}

		return $row;
	}

	/**
	 * @return string|string[]|null
	 */
	protected function getPaymentIdent(?int $paymentId = null)
	{
		if ($this->cPayments === null) {
			$this->cPayments = [];

			foreach ($this->em->createQueryBuilder()->select('p.id, p.ident')
				         ->from(Payment::class, 'p')
				         ->getQuery()->getArrayResult() as $row) {
				$this->cPayments[$row['id']] = $row['ident'];
			}
		}

		return $paymentId ? ($this->cPayments[$paymentId] ?? null) : $this->cPayments;
	}
}
