<?php declare(strict_types = 1);

namespace EshopStatistics\AdminModule\Model;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Contributte\Translation\Translator;
use Core\Model\Helpers\BaseEntityService;
use EshopOrders\Model\Entities\Order;
use EshopStock\Model\Entities\SupplyProduct;
use Nette\DI\Container;
use Nette\Utils\DateTime;

/**
 * @method Order|null getReference($id)
 * @method Order[] getAll()
 * @method Order|null get($id)
 */
class SupplierStatistics extends BaseEntityService
{
	protected int $stockId;

	public function __construct(
		protected Translator $translator,
		protected Container  $container,
	)
	{
		$this->stockId = (int) $this->container->getParameters()['eshopStock']['defaultStockId'];
	}

	// -------------------- statistiky dodavatelu - podle poctu dodaneho zbozi ve skladu ----------------------------------------------

	public function getBySupplier(string $from, string $to): array
	{
		if (!class_exists(SupplyProduct::class)) {
			return [];
		}

		$fromDate = $from !== '' && $from !== '0' ? new DateTime($from) : (new DateTime)->modify('-1 year');
		$toDate   = $to !== '' && $to !== '0' ? new DateTime($to) : (new DateTime);

		$fromDate->setTime(0, 0, 0);
		$toDate->setTime(23, 59, 59);

		//celkova suma vsech dodavatelu
		$qbSum = $this->em->createQueryBuilder();
		$qbSum->select('SUM(sp1.purchasePrice) AS price_all_total')
			->from(SupplyProduct::class, 'sp1')
			->join('sp1.supply', 's1')
			->join('s1.stock', 'st1')
			->andWhere($qbSum->expr()->eq('st1.id', $this->stockId))
			->andWhere($qbSum->expr()->isNull('sp1.writtenOffDate'))
			->andWhere('s1.dateSupply >= :from')->andWhere('s1.dateSupply <= :to');
		$parameters = new ArrayCollection([new Parameter('from', $fromDate), new Parameter('to', $toDate)]);
		$qbSum->setParameters($parameters);
		$priceSumTotal = (float) $qbSum->getQuery()->getSingleScalarResult();

		$result = [];

		//pro jednotlive dodavatele
		$qb = $this->em->createQueryBuilder();
		$qb->select('supp1.id AS supplier_id, supp1.name AS supplier_name, COUNT(DISTINCT(s1.id)) AS count_total, SUM(sp1.purchasePrice) AS price_total')
			->from(SupplyProduct::class, 'sp1')
			->join('sp1.supply', 's1')
			->join('s1.stock', 'st1')
			->leftJoin('s1.supplier', 'supp1')
			->andWhere($qb->expr()->eq('st1.id', $this->stockId))
			->andWhere($qb->expr()->isNull('sp1.writtenOffDate'))
			->andWhere('s1.dateSupply >= :from')->andWhere('s1.dateSupply <= :to')
			->addGroupBy('supp1.id');

		$qb->setParameters($parameters);

		foreach ($qb->getQuery()->getResult() as $row) {

			if ($row['supplier_id']) {
				$key  = $row['supplier_id'];
				$name = $row['supplier_name'];
			} else {
				$key  = '';
				$name = $this->translator->translate('eshopStatistics.supplierStatistics.nameUndef');
			}
			$row['price_total'] = floatval($row['price_total']);

			if (!array_key_exists($key, $result)) {
				$result[$key] = $this->createRowBySupplier($key, $name);
			}

			$result[$key]['count_total'] = $row['count_total'];
			$result[$key]['price_total'] = $row['price_total'];

			/* Pro razeni pouzijeme sloupec zvlast, kde je integer se stonasobnou hodnotou ceny
			 * datagrid totiz pro razeni pouziva keysort, a z floatu udela string, ktery pak seradi pres SORT_REGULAR  a vyjdou nesmysly.
			 * Viz ArrayDataSource->sort()
			 * pokud bychom radili jen podle intval, tak razeni nebude spravne pokud se dve cisla budou lisit jen v desetinach
			 */
			$result[$key]['price_sort'] = intval($row['price_total'] * 100); //
		}

		foreach ($result as $key => $row) {
			$result[$key]['price_percent'] = round(100 * $row['price_total'] / ($priceSumTotal ?: 1), 1);
		}

		return $result;
	}

	protected function createRowBySupplier(int $key, string $supplierName): array
	{
		return [
			'supplier_id'   => $key,
			'supplier_name' => $supplierName,
			'count_total'   => 0, // kolikrat jsme od dodavatele brali zakazku
			'price_total'   => 0, // celkova cena zakazek od dodavatele
			'price_sort'    => 0, // kolikrat jsme od dodavatele brali zakazku
			'price_percent' => 0,
		];
	}

}
