<?php declare(strict_types = 1);

namespace EshopStatistics\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use DateTimeInterface;
use Exception;
use Nette\Utils\DateTime;
use Nette\Utils\Json;
use Tracy\Debugger;

class ZboziItemReportService
{
	public function __construct(
		protected EntityManagerDecorator $em,
		protected Sites                  $sites,
	)
	{
	}

	public function downloadLastMonth(string $monthModifier = '-1 month'): void
	{
		$now = new DateTime();

		$from = (clone $now)->modify($monthModifier);
		$to   = (clone $now)->modify($monthModifier);

		$from = $from->modify('first day of this month')
			->setTime(0, 0, 0);
		$to   = $to->modify('last day of this month')
			->setTime(23, 59, 59);

		foreach ($this->sites->getSites() as $site) {
			$this->downloadReport($site->getIdent(), $from, $to);
		}
	}

	public function downloadReport(string $siteIdent, DateTimeInterface $from, DateTimeInterface $to): void
	{
		$exist = $this->em->getConnection()->fetchOne('SELECT COUNT(*) 
				FROM eshop_statistics__zbozi_report 
				WHERE site_ident = :siteIdent AND `from` = :from AND `to` = :to
				LIMIT 1', [
			':siteIdent' => $siteIdent,
			':from'      => $from->format('Y-m-d H:i:s'),
			':to'        => $to->format('Y-m-d H:i:s'),
		]);

		if ($exist) {
			return;
		}

		$username = EshopStatisticsConfig::loadScalar('zboziApi.' . $siteIdent . '.username');
		$password = EshopStatisticsConfig::loadScalar('zboziApi.' . $siteIdent . '.password');

		if (!$username || !$password) {
			return;
		}

		$authorization = base64_encode("$username:$password");

		$requestData = $this->getRequestId($siteIdent, $authorization, $from, $to);

		if (!$requestData) {
			return;
		}

		sleep(25);

		try {
			$curl = curl_init();

			curl_setopt_array($curl, [
				CURLOPT_URL            => 'https://api.zbozi.cz/v1/shop/statistics/item/json?requestId=' . $requestData['requestId'],
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_ENCODING       => '',
				CURLOPT_MAXREDIRS      => 10,
				CURLOPT_TIMEOUT        => 0,
				CURLOPT_FOLLOWLOCATION => true,
				CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
				CURLOPT_CUSTOMREQUEST  => 'GET',
				CURLOPT_HEADER         => false,
				CURLOPT_HTTPHEADER     => [
					'Authorization: Basic ' . $authorization,
				],
			]);

			$response = curl_exec($curl);
			$httpCode = (int) curl_getinfo($curl, CURLINFO_HTTP_CODE);
			curl_close($curl);

			if ($httpCode === 200) {
				$data = Json::decode((string) $response, Json::FORCE_ARRAY);

				foreach ($data['data'] ?? [] as $row) {
					if (!$row['itemId']) {
						continue;
					}

					$this->em->getConnection()->insert('eshop_statistics__zbozi_item_report', [
						'zbozi_report_id'   => $requestData['entityId'],
						'item_id'           => (int) $row['itemId'],
						'item_name'         => $row['itemTitle'],
						'views'             => (int) $this->getTotalValueFromFieldResponse($row['views']),
						'clicks'            => (int) $this->getTotalValueFromFieldResponse($row['clicks']),
						'costs'             => (int) round($this->getTotalValueFromFieldResponse($row['cost']) * 100, 0),
						'conversions'       => (int) $this->getTotalValueFromFieldResponse($row['conversions']),
						'conversions_value' => (int) round($this->getTotalValueFromFieldResponse($row['conversionsValue']) * 100, 0),
					]);
				}
			}
		} catch (Exception $e) {
			Debugger::log($e, 'zbozi-api');
		}
	}

	protected function getTotalValueFromFieldResponse(array $row): float
	{
		return (float) ($row['categoryListing'] ?? 0)
			+ (float) ($row['categorySearch'] ?? 0)
			+ (float) ($row['productDetail'] ?? 0)
			+ (float) ($row['search'] ?? 0)
			+ (float) ($row['topProductDetail'] ?? 0);
	}

	/**
	 * @return array{
	 *     requestId: string,
	 *     entityId: int
	 * }|null
	 */
	protected function getRequestId(string $siteIdent, string $authorization, DateTimeInterface $from, DateTimeInterface $to): ?array
	{
		$curl = curl_init();

		try {
			curl_setopt_array($curl, [
				CURLOPT_URL            => 'https://api.zbozi.cz/v1/shop/statistics/item?timestampFrom=' . $from->getTimestamp() . '&timestampTo=' . $to->getTimestamp() . '&dataFormat=json',
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_ENCODING       => '',
				CURLOPT_MAXREDIRS      => 10,
				CURLOPT_TIMEOUT        => 0,
				CURLOPT_FOLLOWLOCATION => true,
				CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
				CURLOPT_CUSTOMREQUEST  => 'POST',
				CURLOPT_HEADER         => false,
				CURLOPT_HTTPHEADER     => [
					'Authorization: Basic ' . $authorization,
				],
			]);

			$response = curl_exec($curl);
			$httpCode = (int) curl_getinfo($curl, CURLINFO_HTTP_CODE);
			curl_close($curl);

			if ($httpCode === 201) {
				$data = Json::decode((string) $response, Json::FORCE_ARRAY);

				$requestId = $data['data']['requestId'] ?? null;

				if (!$requestId) {
					throw new Exception('Zbozi API error: request ID not found');
				}

				$this->em->getConnection()->insert('eshop_statistics__zbozi_report', [
					'site_ident' => $siteIdent,
					'request_id' => $requestId,
					'`from`'     => $from->format('Y-m-d H:i:s'),
					'`to`'       => $to->format('Y-m-d H:i:s'),
				]);

				return [
					'requestId' => (string) $requestId,
					'entityId'  => (int) $this->em->getConnection()->lastInsertId(),
				];
			}

			Debugger::log('Zbozi API error: ' . $response, 'zbozi-api');
		} catch (Exception $e) {
			Debugger::log($e, 'zbozi-api');
		}

		return null;
	}
}
